/*
 * Decompiled with CFR 0.152.
 */
package org.scijava.launcher;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Enumeration;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

public final class Archives {
    private static final String TAR_PATTERN = ".*\\.tar(\\.[a-zA-Z0-9]*)?$";

    private Archives() {
    }

    public static Future<Void> unpack(File file, File destDir, Consumer<String> outputConsumer) {
        if (file.getName().endsWith(".zip")) {
            return Archives.unzip(file, destDir, outputConsumer);
        }
        if (file.getName().matches(TAR_PATTERN)) {
            return Archives.untar(file, destDir, outputConsumer);
        }
        throw new IllegalArgumentException("Cannot unpack unsupported file: " + file);
    }

    public static Future<Void> unzip(File file, File destDir, Consumer<String> outputConsumer) {
        ExecutorService executor = Executors.newSingleThreadExecutor();
        return executor.submit(() -> {
            Path destPath = destDir.toPath().normalize();
            try (ZipFile zipFile = new ZipFile(file);){
                Enumeration<? extends ZipEntry> zipEntries = zipFile.entries();
                while (zipEntries.hasMoreElements()) {
                    ZipEntry entry = zipEntries.nextElement();
                    Path entryPath = destPath.resolve(entry.getName()).normalize();
                    if (!entryPath.startsWith(destPath)) {
                        throw new IOException("Entry is outside of the target dir: " + entry.getName());
                    }
                    if (outputConsumer != null) {
                        outputConsumer.accept(entry.getName());
                    }
                    if (entry.isDirectory()) {
                        Files.createDirectories(entryPath, new FileAttribute[0]);
                        continue;
                    }
                    Files.createDirectories(entryPath.getParent(), new FileAttribute[0]);
                    Files.copy(zipFile.getInputStream(entry), entryPath, StandardCopyOption.REPLACE_EXISTING);
                }
            }
            finally {
                executor.shutdown();
            }
            return null;
        });
    }

    public static Future<Void> untar(File file, File destDir, Consumer<String> outputConsumer) {
        ExecutorService executor = Executors.newSingleThreadExecutor();
        return executor.submit(() -> {
            Process p = new ProcessBuilder("tar", "xvf", file.getAbsolutePath(), "-C", destDir.getAbsolutePath()).start();
            StringBuilder errorOutput = new StringBuilder();
            AtomicBoolean shouldStop = new AtomicBoolean(false);
            Thread stdoutThread = new Thread(() -> {
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(p.getInputStream()));){
                    Archives.readLines(reader, shouldStop, outputConsumer);
                }
                catch (IOException e) {
                    Archives.appendException(errorOutput, e);
                }
            });
            Thread stderrThread = new Thread(() -> {
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(p.getErrorStream()));){
                    Archives.readLines(reader, shouldStop, line -> errorOutput.append((String)line).append("\n"));
                }
                catch (IOException e) {
                    Archives.appendException(errorOutput, e);
                }
            });
            stdoutThread.start();
            stderrThread.start();
            try {
                int exitCode = p.waitFor();
                if (exitCode != 0) {
                    throw new IOException("tar command failed with exit code: " + exitCode + "\n\n" + errorOutput);
                }
            }
            catch (InterruptedException e) {
                shouldStop.set(true);
                p.destroy();
                try {
                    if (!p.waitFor(1L, TimeUnit.SECONDS)) {
                        p.destroyForcibly();
                    }
                }
                catch (InterruptedException ie) {
                    Archives.appendException(errorOutput, ie);
                }
                stdoutThread.interrupt();
                stderrThread.interrupt();
                try {
                    stdoutThread.join(500L);
                    stderrThread.join(500L);
                }
                catch (InterruptedException ie) {
                    errorOutput.append("Interrupted while waiting for I/O threads to terminate\n");
                }
                Thread.currentThread().interrupt();
                throw new IOException("Untar operation was interrupted\n" + errorOutput, e);
            }
            finally {
                executor.shutdownNow();
            }
            return null;
        });
    }

    private static void readLines(BufferedReader reader, AtomicBoolean shouldStop, Consumer<String> lineHandler) throws IOException {
        String line;
        while (!shouldStop.get() && !Thread.currentThread().isInterrupted() && (line = reader.readLine()) != null) {
            if (lineHandler == null) continue;
            lineHandler.accept(line);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void appendException(StringBuilder sb, Throwable t) {
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        t.printStackTrace(pw);
        StringBuilder stringBuilder = sb;
        synchronized (stringBuilder) {
            sb.append("\n").append(sw).append("\n");
        }
    }
}

