/*
 * Decompiled with CFR 0.152.
 */
package org.apposed.appose.mamba;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.List;
import java.util.UUID;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.apposed.appose.mamba.FileDownloader;
import org.apposed.appose.mamba.MambaInstallerUtils;

class Mamba {
    final String mambaCommand;
    private final String rootdir;
    private BiConsumer<Long, Long> mambaDownloadProgressConsumer;
    private Consumer<String> outputConsumer;
    private Consumer<String> errorConsumer;
    private static final Path MICROMAMBA_RELATIVE_PATH = Mamba.isWindowsOS() ? Paths.get("Library", "bin", "micromamba.exe") : Paths.get("bin", "micromamba");
    public static final String BASE_PATH = Paths.get(System.getProperty("user.home"), ".local", "share", "appose", "micromamba").toString();
    public static final String MICROMAMBA_URL = "https://micro.mamba.pm/api/micromamba/" + Mamba.microMambaPlatform() + "/latest";
    public static final String ERR_STREAM_UUUID = UUID.randomUUID().toString();

    private static String microMambaPlatform() {
        String osName = System.getProperty("os.name");
        if (osName.startsWith("Windows")) {
            osName = "Windows";
        }
        String osArch = System.getProperty("os.arch");
        switch (osName + "|" + osArch) {
            case "Linux|amd64": {
                return "linux-64";
            }
            case "Linux|aarch64": {
                return "linux-aarch64";
            }
            case "Linux|ppc64le": {
                return "linux-ppc64le";
            }
            case "Mac OS X|x86_64": {
                return "osx-64";
            }
            case "Mac OS X|aarch64": {
                return "osx-arm64";
            }
            case "Windows|amd64": {
                return "win-64";
            }
        }
        return null;
    }

    private void updateMambaDownloadProgress(long current, long total) {
        if (this.mambaDownloadProgressConsumer != null) {
            this.mambaDownloadProgressConsumer.accept(current, total);
        }
    }

    private void updateOutputConsumer(String str) {
        if (this.outputConsumer != null) {
            this.outputConsumer.accept(str == null ? "" : str);
        }
    }

    private void updateErrorConsumer(String str) {
        if (this.errorConsumer != null) {
            this.errorConsumer.accept(str == null ? "" : str);
        }
    }

    private ProcessBuilder getBuilder(boolean isInheritIO) {
        ProcessBuilder builder = new ProcessBuilder(new String[0]).directory(new File(this.rootdir));
        if (isInheritIO) {
            builder.inheritIO();
        }
        return builder;
    }

    public Mamba() {
        this(BASE_PATH);
    }

    public Mamba(String rootdir) {
        this.rootdir = rootdir == null ? BASE_PATH : rootdir;
        this.mambaCommand = Paths.get(this.rootdir, new String[0]).resolve(MICROMAMBA_RELATIVE_PATH).toAbsolutePath().toString();
    }

    public boolean isMambaInstalled() {
        try {
            this.getVersion();
            return true;
        }
        catch (IOException | InterruptedException e) {
            return false;
        }
    }

    private void checkMambaInstalled() {
        if (!this.isMambaInstalled()) {
            throw new IllegalStateException("Micromamba is not installed");
        }
    }

    public void setMambaDownloadProgressConsumer(BiConsumer<Long, Long> consumer) {
        this.mambaDownloadProgressConsumer = consumer;
    }

    public void setOutputConsumer(Consumer<String> consumer) {
        this.outputConsumer = consumer;
    }

    public void setErrorConsumer(Consumer<String> consumer) {
        this.errorConsumer = consumer;
    }

    private File downloadMicromamba() throws IOException, InterruptedException, URISyntaxException {
        File tempFile = File.createTempFile("micromamba", ".tar.bz2");
        tempFile.deleteOnExit();
        URL website = MambaInstallerUtils.redirectedURL(new URL(MICROMAMBA_URL));
        long size = MambaInstallerUtils.getFileSize(website);
        Thread currentThread = Thread.currentThread();
        IOException[] ioe = new IOException[]{null};
        InterruptedException[] ie = new InterruptedException[]{null};
        Thread dwnldThread = new Thread(() -> {
            try (ReadableByteChannel rbc = Channels.newChannel(website.openStream());
                 FileOutputStream fos = new FileOutputStream(tempFile);){
                new FileDownloader(rbc, fos).call(currentThread);
            }
            catch (IOException e) {
                ioe[0] = e;
            }
            catch (InterruptedException e) {
                ie[0] = e;
            }
        });
        dwnldThread.start();
        while (dwnldThread.isAlive()) {
            Thread.sleep(20L);
            this.updateMambaDownloadProgress(tempFile.length(), size);
        }
        if (ioe[0] != null) {
            throw ioe[0];
        }
        if (ie[0] != null) {
            throw ie[0];
        }
        if (tempFile.length() < size) {
            throw new IOException("Error downloading micromamba from: " + MICROMAMBA_URL);
        }
        return tempFile;
    }

    private void decompressMicromamba(File tempFile) throws IOException, InterruptedException {
        boolean executableSet;
        File tempTarFile = File.createTempFile("micromamba", ".tar");
        tempTarFile.deleteOnExit();
        MambaInstallerUtils.unBZip2(tempFile, tempTarFile);
        File mambaBaseDir = new File(this.rootdir);
        if (!mambaBaseDir.isDirectory() && !mambaBaseDir.mkdirs()) {
            throw new IOException("Failed to create Micromamba default directory " + mambaBaseDir.getParentFile().getAbsolutePath() + ". Please try installing it in another directory.");
        }
        MambaInstallerUtils.unTar(tempTarFile, mambaBaseDir);
        File mmFile = new File(this.mambaCommand);
        if (!mmFile.exists()) {
            throw new IOException("Expected micromamba binary is missing: " + this.mambaCommand);
        }
        if (!mmFile.canExecute() && !(executableSet = new File(this.mambaCommand).setExecutable(true))) {
            throw new IOException("Cannot set file as executable due to missing permissions, please do it manually: " + this.mambaCommand);
        }
    }

    public void installMicromamba() throws IOException, InterruptedException, URISyntaxException {
        if (this.isMambaInstalled()) {
            return;
        }
        this.decompressMicromamba(this.downloadMicromamba());
    }

    private static List<String> getBaseCommand() {
        ArrayList<String> cmd = new ArrayList<String>();
        if (Mamba.isWindowsOS()) {
            cmd.addAll(Arrays.asList("cmd.exe", "/c"));
        }
        return cmd;
    }

    public void updateIn(File envDir, String ... args) throws IOException, InterruptedException {
        this.checkMambaInstalled();
        ArrayList<String> cmd = new ArrayList<String>(Arrays.asList("update", "--prefix", envDir.getAbsolutePath()));
        cmd.addAll(Arrays.asList(args));
        if (!cmd.contains("--yes") && !cmd.contains("-y")) {
            cmd.add("--yes");
        }
        this.runMamba(cmd.toArray(new String[0]));
    }

    public void createWithYaml(File envDir, String envYaml) throws IOException, InterruptedException {
        this.checkMambaInstalled();
        this.runMamba("env", "create", "--prefix", envDir.getAbsolutePath(), "-f", envYaml, "-y", "-vv");
    }

    public String getVersion() throws IOException, InterruptedException {
        List<String> cmd = Mamba.getBaseCommand();
        if (this.mambaCommand.contains(" ") && Mamba.isWindowsOS()) {
            cmd.add(Mamba.surroundWithQuotes(Arrays.asList(Mamba.coverArgWithDoubleQuotes(this.mambaCommand), "--version")));
        } else {
            cmd.addAll(Arrays.asList(Mamba.coverArgWithDoubleQuotes(this.mambaCommand), "--version"));
        }
        Process process = this.getBuilder(false).command(cmd).start();
        if (process.waitFor() != 0) {
            throw new RuntimeException("Error getting Micromamba version");
        }
        return new BufferedReader(new InputStreamReader(process.getInputStream())).readLine();
    }

    public void runMamba(boolean isInheritIO, String ... args) throws RuntimeException, IOException, InterruptedException {
        int processResult;
        this.checkMambaInstalled();
        Thread mainThread = Thread.currentThread();
        SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
        List<String> cmd = Mamba.getBaseCommand();
        ArrayList<String> argsList = new ArrayList<String>();
        argsList.add(Mamba.coverArgWithDoubleQuotes(this.mambaCommand));
        argsList.addAll(Arrays.stream(args).map(aa -> {
            if (aa.contains(" ") && Mamba.isWindowsOS()) {
                return Mamba.coverArgWithDoubleQuotes(aa);
            }
            return aa;
        }).collect(Collectors.toList()));
        boolean containsSpaces = argsList.stream().anyMatch(aa -> aa.contains(" "));
        if (!containsSpaces || !Mamba.isWindowsOS()) {
            cmd.addAll(argsList);
        } else {
            cmd.add(Mamba.surroundWithQuotes(argsList));
        }
        ProcessBuilder builder = this.getBuilder(isInheritIO).command(cmd);
        Process process = builder.start();
        this.updateOutputConsumer(sdf.format(Calendar.getInstance().getTime()) + " -- STARTING INSTALLATION" + System.lineSeparator());
        long updatePeriod = 300L;
        Thread outputThread = new Thread(() -> {
            try (InputStream inputStream = process.getInputStream();
                 InputStream errStream = process.getErrorStream();){
                byte[] buffer = new byte[1024];
                StringBuilder processBuff = new StringBuilder();
                StringBuilder errBuff = new StringBuilder();
                String processChunk = "";
                String errChunk = "";
                long t0 = System.currentTimeMillis();
                while (process.isAlive() || inputStream.available() > 0) {
                    int newLineIndex;
                    if (!mainThread.isAlive()) {
                        process.destroyForcibly();
                        return;
                    }
                    if (inputStream.available() > 0) {
                        processBuff.append(new String(buffer, 0, inputStream.read(buffer)));
                        while ((newLineIndex = processBuff.indexOf(System.lineSeparator())) != -1) {
                            processChunk = processChunk + sdf.format(Calendar.getInstance().getTime()) + " -- " + processBuff.substring(0, newLineIndex + 1).trim() + System.lineSeparator();
                            processBuff.delete(0, newLineIndex + 1);
                        }
                    }
                    if (errStream.available() > 0) {
                        errBuff.append(new String(buffer, 0, errStream.read(buffer)));
                        while ((newLineIndex = errBuff.indexOf(System.lineSeparator())) != -1) {
                            errChunk = errChunk + ERR_STREAM_UUUID + errBuff.substring(0, newLineIndex + 1).trim() + System.lineSeparator();
                            errBuff.delete(0, newLineIndex + 1);
                        }
                    }
                    Thread.sleep(60L);
                    if (System.currentTimeMillis() - t0 <= updatePeriod) continue;
                    this.updateOutputConsumer(processChunk);
                    processChunk = "";
                    errChunk = "";
                    t0 = System.currentTimeMillis();
                }
                if (inputStream.available() > 0) {
                    processBuff.append(new String(buffer, 0, inputStream.read(buffer)));
                    processChunk = processChunk + sdf.format(Calendar.getInstance().getTime()) + " -- " + processBuff.toString().trim();
                }
                if (errStream.available() > 0) {
                    errBuff.append(new String(buffer, 0, errStream.read(buffer)));
                    errChunk = errChunk + ERR_STREAM_UUUID + errBuff.toString().trim();
                }
                this.updateErrorConsumer(errChunk);
                this.updateOutputConsumer(processChunk + System.lineSeparator() + sdf.format(Calendar.getInstance().getTime()) + " -- TERMINATED PROCESS\n");
            }
            catch (IOException | InterruptedException e) {
                e.printStackTrace();
            }
        });
        outputThread.start();
        try {
            processResult = process.waitFor();
        }
        catch (InterruptedException ex) {
            throw new InterruptedException("Mamba process stopped. The command being executed was: " + cmd);
        }
        outputThread.join();
        if (processResult != 0) {
            throw new RuntimeException("Exit code " + processResult + " from command execution: " + builder.command());
        }
    }

    public void runMamba(String ... args) throws RuntimeException, IOException, InterruptedException {
        this.checkMambaInstalled();
        this.runMamba(false, args);
    }

    private static String coverArgWithDoubleQuotes(String arg) {
        String[] specialChars;
        for (String schar : specialChars = new String[]{" "}) {
            if (arg.startsWith("\"") && arg.endsWith("\"") || !arg.contains(schar) || !Mamba.isWindowsOS()) continue;
            return "\"" + arg + "\"";
        }
        return arg;
    }

    private static String surroundWithQuotes(List<String> args) {
        String arg = "\"";
        for (String aa : args) {
            arg = arg + aa + " ";
        }
        arg = arg.substring(0, arg.length() - 1);
        arg = arg + "\"";
        return arg;
    }

    private static boolean isWindowsOS() {
        return System.getProperty("os.name").startsWith("Windows");
    }
}

