/*
 * Decompiled with CFR 0.152.
 */
package fiji.plugin.trackmate.util.cli.condapath;

import ij.IJ;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class CondaDetector {
    private static CondaInfo cachedInfo = null;
    private static long cacheTimestamp = 0L;
    private static final long CACHE_TIMEOUT_MS = 60000L;

    public static CondaInfo detect() throws CondaNotFoundException {
        long now = System.currentTimeMillis();
        if (cachedInfo != null && now - cacheTimestamp < 60000L) {
            return cachedInfo;
        }
        CondaInfo info = CondaDetector.detectCondaInfo();
        if (info != null) {
            cachedInfo = info;
            cacheTimestamp = now;
            return info;
        }
        throw new CondaNotFoundException("Could not find conda installation. Please ensure conda is installed and initialized.\nSuggestions:\n  1. Install Anaconda or Miniconda\n  2. Run 'conda init' in your terminal\n  3. Restart your application");
    }

    public static void clearCache() {
        cachedInfo = null;
        cacheTimestamp = 0L;
    }

    private static CondaInfo detectCondaInfo() {
        IJ.log((String)"Starting conda detection...");
        CondaInfo info = CondaDetector.detectFromEnvironment();
        if (info != null) {
            return info;
        }
        IJ.log((String)"");
        if (CondaDetector.isMac() || CondaDetector.isLinux()) {
            info = CondaDetector.detectFromShellConfig();
            if (info != null) {
                return info;
            }
        } else {
            IJ.log((String)"Method 2: Parsing shell configuration files...");
            IJ.log((String)"  Skipped (only applicable on macOS/Linux)");
        }
        IJ.log((String)"");
        info = CondaDetector.detectFromPath();
        if (info != null) {
            return info;
        }
        IJ.log((String)"");
        info = CondaDetector.detectFromCommonLocations();
        if (info != null) {
            return info;
        }
        IJ.log((String)"");
        IJ.log((String)"Failed to detect conda installation");
        return null;
    }

    private static CondaInfo detectFromEnvironment() {
        IJ.log((String)"Method 1: Checking CONDA_EXE environment variable...");
        String condaExe = System.getenv("CONDA_EXE");
        if (condaExe == null || condaExe.isEmpty()) {
            condaExe = System.getenv("_CONDA_EXE");
        }
        if (condaExe != null && new File(condaExe).exists() && (condaExe = CondaDetector.resolveRealExecutable(condaExe)) != null) {
            String rootPrefix = CondaDetector.deriveRootPrefix(condaExe);
            String version = CondaDetector.getCondaVersion(condaExe);
            if (rootPrefix != null && version != null) {
                IJ.log((String)("  Found conda via CONDA_EXE: " + condaExe));
                return new CondaInfo(condaExe, rootPrefix, version);
            }
        }
        IJ.log((String)"  CONDA_EXE not set or invalid");
        return null;
    }

    private static CondaInfo detectFromShellConfig() {
        String[] configFiles;
        IJ.log((String)"Method 2: Parsing shell configuration files...");
        String home = System.getProperty("user.home");
        for (String configFile : configFiles = new String[]{".zshrc", ".bash_profile", ".bashrc", ".profile", ".config/fish/config.fish"}) {
            Path configPath = Paths.get(home, configFile);
            if (!Files.exists(configPath, new LinkOption[0])) continue;
            try {
                String content = new String(Files.readAllBytes(configPath));
                if (!content.contains("conda.sh") && !content.contains("mamba.sh") && !content.contains("conda initialize") && !content.contains("conda") && !content.contains("mamba") && !content.contains("micromamba")) continue;
                IJ.log((String)("  Checking: " + configFile));
                String condaExe = CondaDetector.extractCondaExeFromConfig(content, configPath);
                if (condaExe == null || !new File(condaExe).exists()) continue;
                condaExe = CondaDetector.resolveRealExecutable(condaExe);
                String rootPrefix = CondaDetector.deriveRootPrefix(condaExe);
                String version = CondaDetector.getCondaVersion(condaExe);
                if (rootPrefix != null && version != null) {
                    IJ.log((String)("  Found conda by parsing: " + configFile));
                    return new CondaInfo(condaExe, rootPrefix, version);
                }
                IJ.log((String)("  Found executable but failed to validate: " + condaExe));
            }
            catch (IOException e) {
                IJ.log((String)("  Could not read: " + configFile));
            }
        }
        IJ.log((String)"  No conda found in shell config files");
        return null;
    }

    private static String extractCondaExeFromConfig(String content, Path configPath) {
        String binDir;
        String microExe;
        String mambaExe;
        String condaExe;
        String condaExe2;
        Pattern pattern = Pattern.compile("__(conda|mamba)_setup=[\"']\\$\\([\"']([^\"']+/(conda|mamba|micromamba))[\"']");
        Matcher matcher = pattern.matcher(content);
        if (matcher.find() && new File(condaExe2 = matcher.group(2)).exists()) {
            IJ.log((String)("    Found via setup variable: " + condaExe2));
            return condaExe2;
        }
        pattern = Pattern.compile("export (CONDA_EXE|MAMBA_EXE|MAMBA_ROOT_PREFIX)=['\"]?([^'\"\\s]+)['\"]?");
        matcher = pattern.matcher(content);
        while (matcher.find()) {
            String varName = matcher.group(1);
            Object exe = matcher.group(2);
            if (varName.equals("MAMBA_ROOT_PREFIX")) {
                exe = (String)exe + "/bin/micromamba";
            }
            if (!new File((String)exe).exists()) continue;
            IJ.log((String)("    Found via " + varName + ": " + (String)exe));
            return exe;
        }
        pattern = Pattern.compile("[.\\s][\\s\"']*([^\"']+)/(etc/profile\\.d/(conda|mamba|micromamba)\\.sh)[\"']");
        matcher = pattern.matcher(content);
        if (matcher.find()) {
            String rootPrefix = matcher.group(1);
            condaExe = rootPrefix + "/bin/conda";
            if (new File(condaExe).exists()) {
                IJ.log((String)("    Found via profile.d: " + condaExe));
                return condaExe;
            }
            mambaExe = rootPrefix + "/bin/mamba";
            if (new File(mambaExe).exists()) {
                IJ.log((String)("    Found via profile.d: " + mambaExe));
                return mambaExe;
            }
            microExe = rootPrefix + "/bin/micromamba";
            if (new File(microExe).exists()) {
                IJ.log((String)("    Found via profile.d: " + microExe));
                return microExe;
            }
        }
        if ((matcher = (pattern = Pattern.compile("export PATH=[\"']([^\"':]+/(mambaforge|miniconda|anaconda|miniforge|micromamba)[^\"':]*)[\"':]")).matcher(content)).find()) {
            binDir = matcher.group(1);
            condaExe = binDir + "/conda";
            if (new File(condaExe).exists()) {
                IJ.log((String)("    Found via PATH export: " + condaExe));
                return condaExe;
            }
            mambaExe = binDir + "/mamba";
            if (new File(mambaExe).exists()) {
                IJ.log((String)("    Found via PATH export: " + mambaExe));
                return mambaExe;
            }
            microExe = binDir + "/micromamba";
            if (new File(microExe).exists()) {
                IJ.log((String)("    Found via PATH export: " + microExe));
                return microExe;
            }
        }
        pattern = Pattern.compile("(?:path[+=]+|export PATH=)[^\"']*[\"']([^\"':]+/bin)[\"':]");
        matcher = pattern.matcher(content);
        while (matcher.find()) {
            binDir = matcher.group(1);
            condaExe = binDir + "/conda";
            if (new File(condaExe).exists()) {
                IJ.log((String)("    Found via path assignment: " + condaExe));
                return condaExe;
            }
            mambaExe = binDir + "/mamba";
            if (new File(mambaExe).exists()) {
                IJ.log((String)("    Found via path assignment: " + mambaExe));
                return mambaExe;
            }
            microExe = binDir + "/micromamba";
            if (!new File(microExe).exists()) continue;
            IJ.log((String)("    Found via path assignment: " + microExe));
            return microExe;
        }
        return null;
    }

    private static CondaInfo detectFromPath() {
        IJ.log((String)"Method 3: Searching system PATH...");
        String condaExe = CondaDetector.findInPath("conda");
        if (condaExe != null) {
            condaExe = CondaDetector.resolveRealExecutable(condaExe);
            String rootPrefix = CondaDetector.deriveRootPrefix(condaExe);
            String version = CondaDetector.getCondaVersion(condaExe);
            if (rootPrefix != null && version != null) {
                IJ.log((String)("  Found conda in PATH: " + condaExe));
                return new CondaInfo(condaExe, rootPrefix, version);
            }
        }
        IJ.log((String)"  Conda not found in PATH");
        return null;
    }

    private static CondaInfo detectFromCommonLocations() {
        IJ.log((String)"Method 4: Checking common installation locations...");
        List<String> locations = CondaDetector.getCommonCondaLocations();
        for (String location : locations) {
            String version;
            String condaExe;
            File dir = new File(location);
            if (!dir.exists() || !dir.isDirectory() || (condaExe = CondaDetector.buildCondaExecutablePath(location)) == null || !new File(condaExe).exists() || (version = CondaDetector.getCondaVersion(condaExe = CondaDetector.resolveRealExecutable(condaExe))) == null) continue;
            IJ.log((String)("  Found conda at: " + location));
            return new CondaInfo(condaExe, location, version);
        }
        IJ.log((String)"  No conda found in common locations");
        return null;
    }

    private static String resolveRealExecutable(String executablePath) {
        if (executablePath == null) {
            return null;
        }
        try {
            File file = new File(executablePath);
            String realPath = file.getCanonicalPath();
            if (!CondaDetector.isWindows() && Files.isSymbolicLink(Paths.get(executablePath, new String[0]))) {
                try {
                    Path resolved = Files.readSymbolicLink(Paths.get(executablePath, new String[0]));
                    if (!resolved.isAbsolute()) {
                        Path parent = Paths.get(executablePath, new String[0]).getParent();
                        resolved = parent.resolve(resolved).normalize();
                    }
                    if (!executablePath.equals(realPath = resolved.toString())) {
                        IJ.log((String)("  Resolved symlink: " + executablePath + " -> " + realPath));
                    }
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            return realPath;
        }
        catch (IOException e) {
            IJ.log((String)("  Could not resolve path: " + executablePath));
            return executablePath;
        }
    }

    private static boolean isMicromambaExecutable(String condaExePath) {
        if (condaExePath != null && condaExePath.toLowerCase().contains("micromamba")) {
            return true;
        }
        try {
            List<String> command = CondaDetector.buildSimpleCommand(condaExePath, "--version");
            ProcessBuilder pb = new ProcessBuilder(command);
            pb.redirectErrorStream(true);
            Process process = pb.start();
            String output = CondaDetector.readProcessOutput(process);
            boolean completed = process.waitFor(5L, TimeUnit.SECONDS);
            if (completed && process.exitValue() == 0 && output != null) {
                return output.toLowerCase().contains("micromamba");
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return false;
    }

    private static String deriveRootPrefix(String condaExePath) {
        try {
            String microRootPrefix;
            Path path = Paths.get(condaExePath, new String[0]).toAbsolutePath().normalize();
            String exeName = path.getFileName().toString();
            if (exeName.contains("micromamba") && (microRootPrefix = CondaDetector.getMicromambaRootPrefix(condaExePath)) != null) {
                IJ.log((String)("  Micromamba root prefix from 'micromamba info': " + microRootPrefix));
                return microRootPrefix;
            }
            if (CondaDetector.isWindows()) {
                String pathStr = path.toString();
                if (pathStr.contains("\\Library\\bin\\conda")) {
                    Path rootPrefix;
                    Path parent = path.getParent();
                    if (parent != null && (parent = parent.getParent()) != null && (rootPrefix = parent.getParent()) != null) {
                        return rootPrefix.toString();
                    }
                } else {
                    Path rootPrefix;
                    Path parent = path.getParent();
                    if (parent != null && (rootPrefix = parent.getParent()) != null) {
                        return rootPrefix.toString();
                    }
                }
            } else {
                Path rootPrefix;
                Path parent = path.getParent();
                if (parent != null && (rootPrefix = parent.getParent()) != null) {
                    if (exeName.contains("micromamba")) {
                        File envsDir = new File(rootPrefix.toFile(), "envs");
                        File pkgsDir = new File(rootPrefix.toFile(), "pkgs");
                        if (!envsDir.exists() && !pkgsDir.exists()) {
                            String[] defaultMicroRoots;
                            String home = System.getProperty("user.home");
                            for (String defaultRoot : defaultMicroRoots = new String[]{home + "/micromamba", home + "/.mamba", home + "/.local/share/mamba"}) {
                                File defaultEnvs = new File(defaultRoot, "envs");
                                if (!defaultEnvs.exists()) continue;
                                IJ.log((String)("  Using default micromamba root: " + defaultRoot));
                                return defaultRoot;
                            }
                            String fallback = home + "/micromamba";
                            IJ.log((String)("  Using fallback micromamba root: " + fallback));
                            return fallback;
                        }
                    }
                    return rootPrefix.toString();
                }
            }
        }
        catch (Exception e) {
            IJ.log((String)("Failed to derive root prefix from " + condaExePath + ": " + e.getMessage()));
        }
        return null;
    }

    private static String getMicromambaRootPrefix(String micromambaPath) {
        try {
            List<String> command = CondaDetector.buildSimpleCommand(micromambaPath, "info");
            ProcessBuilder pb = new ProcessBuilder(command);
            pb.redirectErrorStream(true);
            Process process = pb.start();
            String output = CondaDetector.readProcessOutput(process);
            boolean completed = process.waitFor(10L, TimeUnit.SECONDS);
            if (completed && process.exitValue() == 0 && output != null) {
                String rootPrefix;
                String rootPrefix2;
                Pattern pattern = Pattern.compile("base environment\\s*:\\s*([^\\s]+)");
                Matcher matcher = pattern.matcher(output);
                if (matcher.find() && new File(rootPrefix2 = matcher.group(1)).exists()) {
                    return rootPrefix2;
                }
                Pattern pattern2 = Pattern.compile("root prefix\\s*:\\s*([^\\s]+)");
                Matcher matcher2 = pattern2.matcher(output);
                if (matcher2.find() && new File(rootPrefix = matcher2.group(1)).exists()) {
                    return rootPrefix;
                }
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return null;
    }

    private static String getCondaVersion(String condaExePath) {
        try {
            List<String> command = CondaDetector.buildSimpleCommand(condaExePath, "--version");
            ProcessBuilder pb = new ProcessBuilder(command);
            pb.redirectErrorStream(true);
            Process process = pb.start();
            String output = CondaDetector.readProcessOutput(process);
            boolean completed = process.waitFor(10L, TimeUnit.SECONDS);
            if (completed && process.exitValue() == 0 && output != null) {
                String version = output.trim().replace("conda", "").replace("mamba", "").replace("micromamba", "").trim();
                return version;
            }
        }
        catch (Exception e) {
            IJ.log((String)("Failed to get conda version from " + condaExePath + ": " + e.getMessage()));
        }
        return null;
    }

    private static String findInPath(String executable) {
        block14: {
            String[] stringArray;
            if (CondaDetector.isWindows()) {
                String[] stringArray2 = new String[2];
                stringArray2[0] = "where";
                stringArray = stringArray2;
                stringArray2[1] = executable;
            } else {
                String[] stringArray3 = new String[2];
                stringArray3[0] = "which";
                stringArray = stringArray3;
                stringArray3[1] = executable;
            }
            String[] command = stringArray;
            try {
                String[] lines;
                ProcessBuilder pb = new ProcessBuilder(command);
                pb.redirectErrorStream(true);
                Process process = pb.start();
                String output = CondaDetector.readProcessOutput(process);
                boolean completed = process.waitFor(5L, TimeUnit.SECONDS);
                if (!completed) {
                    IJ.log((String)("  Command timed out searching for '" + executable + "'"));
                    return null;
                }
                if (process.exitValue() != 0) {
                    IJ.log((String)("  '" + executable + "' not found in PATH"));
                    return null;
                }
                if (output == null || output.isEmpty()) break block14;
                for (String line : lines = output.split("\n")) {
                    String path = line.trim();
                    if (path.isEmpty()) continue;
                    if (path.startsWith("INFO:") || path.startsWith("ERROR:") || path.startsWith("WARNING:") || path.contains("Could not find")) {
                        IJ.log((String)("  Skipping error message: " + path));
                        continue;
                    }
                    if (!CondaDetector.isValidPath(path)) {
                        IJ.log((String)("  Skipping invalid path format: " + path));
                        continue;
                    }
                    if (CondaDetector.isWindows() && path.contains("\\Library\\bin\\conda")) {
                        IJ.log((String)("  Skipping helper script: " + path));
                        continue;
                    }
                    if (!new File(path).exists()) {
                        IJ.log((String)("  Skipping non-existent path: " + path));
                        continue;
                    }
                    if (!CondaDetector.isWindows()) {
                        try {
                            path = new File(path).getCanonicalPath();
                        }
                        catch (IOException iOException) {
                            // empty catch block
                        }
                    }
                    return path;
                }
            }
            catch (Exception e) {
                IJ.log((String)("  Error searching PATH: " + e.getMessage()));
            }
        }
        return null;
    }

    private static boolean isValidPath(String path) {
        if (path == null || path.isEmpty()) {
            return false;
        }
        if (path.startsWith("INFO:") || path.startsWith("ERROR:") || path.startsWith("WARNING:") || path.toLowerCase().contains("could not find") || path.toLowerCase().contains("not recognized")) {
            return false;
        }
        if (CondaDetector.isWindows()) {
            if (!(path.contains(":") || path.startsWith("\\\\") || path.contains("\\") || new File(path).exists())) {
                return false;
            }
            String invalidChars = "<>|\u0000";
            for (int i = 0; i < "<>|\u0000".length(); ++i) {
                if (path.indexOf("<>|\u0000".charAt(i)) < 0) continue;
                return false;
            }
        } else if (!path.contains("/") && path.contains(" ") && path.length() > 30) {
            return false;
        }
        return true;
    }

    private static String buildCondaExecutablePath(String installDir) {
        if (CondaDetector.isWindows()) {
            String scriptsPath = Paths.get(installDir, "Scripts", "conda.exe").toString();
            if (new File(scriptsPath).exists()) {
                return scriptsPath;
            }
            scriptsPath = Paths.get(installDir, "Scripts", "mamba.exe").toString();
            if (new File(scriptsPath).exists()) {
                return scriptsPath;
            }
            scriptsPath = Paths.get(installDir, "Scripts", "micromamba.exe").toString();
            if (new File(scriptsPath).exists()) {
                return scriptsPath;
            }
            String condabinPath = Paths.get(installDir, "condabin", "conda.bat").toString();
            if (new File(condabinPath).exists()) {
                return condabinPath;
            }
            scriptsPath = Paths.get(installDir, "Scripts", "conda.bat").toString();
            if (new File(scriptsPath).exists()) {
                return scriptsPath;
            }
        } else {
            String binPath = Paths.get(installDir, "bin", "conda").toString();
            if (new File(binPath).exists()) {
                return binPath;
            }
            binPath = Paths.get(installDir, "bin", "mamba").toString();
            if (new File(binPath).exists()) {
                return binPath;
            }
            binPath = Paths.get(installDir, "bin", "micromamba").toString();
            if (new File(binPath).exists()) {
                return binPath;
            }
        }
        return null;
    }

    private static List<String> getCommonCondaLocations() {
        ArrayList<String> paths = new ArrayList<String>();
        String home = System.getProperty("user.home");
        if (CondaDetector.isWindows()) {
            String[] drives;
            paths.add(home + "\\miniconda3");
            paths.add(home + "\\anaconda3");
            paths.add(home + "\\Miniconda3");
            paths.add(home + "\\Anaconda3");
            paths.add(home + "\\miniforge3");
            paths.add(home + "\\mambaforge");
            paths.add(home + "\\micromamba");
            paths.add(home + "\\AppData\\Local\\miniforge3");
            paths.add(home + "\\AppData\\Local\\mambaforge");
            paths.add(home + "\\AppData\\Local\\miniconda3");
            paths.add(home + "\\AppData\\Local\\anaconda3");
            paths.add(home + "\\AppData\\Local\\micromamba");
            paths.add("C:\\ProgramData\\miniconda3");
            paths.add("C:\\ProgramData\\anaconda3");
            paths.add("C:\\ProgramData\\miniforge3");
            paths.add("C:\\ProgramData\\mambaforge");
            paths.add("C:\\ProgramData\\micromamba");
            paths.add("C:\\tools\\miniconda3");
            paths.add("C:\\tools\\anaconda3");
            paths.add("C:\\tools\\micromamba");
            for (String drive : drives = new String[]{"D:", "E:", "F:"}) {
                paths.add(drive + "\\ProgramData\\miniconda3");
                paths.add(drive + "\\ProgramData\\anaconda3");
                paths.add(drive + "\\ProgramData\\miniforge3");
                paths.add(drive + "\\ProgramData\\micromamba");
            }
        } else if (CondaDetector.isMac()) {
            paths.add(home + "/miniconda3");
            paths.add(home + "/anaconda3");
            paths.add(home + "/miniforge3");
            paths.add(home + "/mambaforge");
            paths.add(home + "/micromamba");
            paths.add(home + "/Applications");
            paths.add("/opt/miniconda3");
            paths.add("/opt/anaconda3");
            paths.add("/opt/conda");
            paths.add("/opt/micromamba");
            paths.add("/usr/local/miniconda3");
            paths.add("/usr/local/anaconda3");
            paths.add("/usr/local/micromamba");
            paths.add("/usr/local/bin");
        } else {
            paths.add(home + "/miniconda3");
            paths.add(home + "/anaconda3");
            paths.add(home + "/miniforge3");
            paths.add(home + "/mambaforge");
            paths.add(home + "/micromamba");
            paths.add(home + "/.local");
            paths.add("/opt/conda");
            paths.add("/opt/miniconda3");
            paths.add("/opt/anaconda3");
            paths.add("/opt/micromamba");
            paths.add("/usr/local/miniconda3");
            paths.add("/usr/local/anaconda3");
            paths.add("/usr/local/micromamba");
        }
        return paths;
    }

    private static List<String> buildSimpleCommand(String condaExePath, String ... args) {
        ArrayList<String> command = new ArrayList<String>();
        if (CondaDetector.isWindows() && !condaExePath.endsWith(".exe")) {
            command.add("cmd.exe");
            command.add("/c");
        }
        command.add(condaExePath);
        command.addAll(Arrays.asList(args));
        return command;
    }

    private static String readProcessOutput(Process process) throws IOException {
        StringBuilder output = new StringBuilder();
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
            String line;
            while ((line = reader.readLine()) != null) {
                if (output.length() > 0) {
                    output.append("\n");
                }
                output.append(line);
            }
        }
        return output.toString();
    }

    private static boolean isWindows() {
        return System.getProperty("os.name").toLowerCase().contains("win");
    }

    private static boolean isMac() {
        String os = System.getProperty("os.name").toLowerCase();
        return os.contains("mac") || os.contains("darwin");
    }

    private static boolean isLinux() {
        return System.getProperty("os.name").toLowerCase().contains("linux");
    }

    private static List<CondaEnvironment> parseEnvironmentsJson(String json) {
        ArrayList<CondaEnvironment> environments = new ArrayList<CondaEnvironment>();
        int envsStart = json.indexOf("\"envs\":");
        if (envsStart != -1) {
            int arrayStart = json.indexOf("[", envsStart);
            int arrayEnd = json.indexOf("]", arrayStart);
            if (arrayStart != -1 && arrayEnd != -1) {
                String[] paths;
                String envsArray = json.substring(arrayStart + 1, arrayEnd);
                for (String path : paths = envsArray.split(",")) {
                    if ((path = path.trim().replace("\"", "").replace("\\\\", "\\")).isEmpty()) continue;
                    File envDir = new File(path);
                    String name = envDir.getName();
                    if (name.equals("miniconda3") || name.equals("anaconda3") || name.equals("mambaforge") || name.equals("miniforge3")) {
                        name = "base";
                    }
                    environments.add(new CondaEnvironment(name, path));
                }
            }
        }
        return environments;
    }

    public static List<CondaEnvironment> findAllEnvironments() throws CondaNotFoundException {
        String condaPath = CondaDetector.detect().getCondaExecutable();
        List<CondaEnvironment> environments = new ArrayList<CondaEnvironment>();
        try {
            ArrayList<String> command = new ArrayList<String>();
            if (CondaDetector.isWindows() && !condaPath.endsWith(".exe")) {
                command.add("cmd.exe");
                command.add("/c");
            }
            command.add(condaPath);
            command.add("env");
            command.add("list");
            command.add("--json");
            ProcessBuilder pb = new ProcessBuilder(command);
            pb.redirectErrorStream(true);
            Process process = pb.start();
            StringBuilder json = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    json.append(line);
                }
            }
            process.waitFor(30L, TimeUnit.SECONDS);
            environments = CondaDetector.parseEnvironmentsJson(json.toString());
        }
        catch (IOException | InterruptedException e) {
            IJ.log((String)("Failed to list conda environments: " + e.getMessage()));
        }
        return environments;
    }

    public static boolean environmentExists(String envName) throws CondaNotFoundException {
        return CondaDetector.findAllEnvironments().stream().anyMatch(env -> env.getName().equals(envName));
    }

    public static String getEnvironmentPath(String envName) throws CondaNotFoundException {
        return CondaDetector.findAllEnvironments().stream().filter(env -> env.getName().equals(envName)).map(CondaEnvironment::getPath).findFirst().orElse(null);
    }

    public static void main(String[] args) {
        CondaDetector.diagnose();
    }

    public static void diagnose() {
        IJ.log((String)"\u2554\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2557");
        IJ.log((String)"\u2551   Conda Detection Diagnosis System     \u2551");
        IJ.log((String)"\u255a\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u255d");
        IJ.log((String)"");
        IJ.log((String)"Starting conda detection process...");
        IJ.log((String)"");
        IJ.log((String)"System Information:");
        IJ.log((String)("  OS:           " + System.getProperty("os.name")));
        IJ.log((String)("  Architecture: " + System.getProperty("os.arch")));
        IJ.log((String)("  User Home:    " + System.getProperty("user.home")));
        IJ.log((String)"");
        IJ.log((String)"Attempting Detection Methods:");
        IJ.log((String)"");
        IJ.log((String)"Method 1: Checking CONDA_EXE environment variable...");
        String condaEnvVar = System.getenv("CONDA_EXE");
        if (condaEnvVar != null) {
            IJ.log((String)("  Found: " + condaEnvVar));
            if (new File(condaEnvVar).exists()) {
                IJ.log((String)"  \u2713 File exists and is accessible");
            } else {
                IJ.log((String)"  \u2717 File does not exist");
            }
        } else {
            IJ.log((String)"  CONDA_EXE not set");
        }
        IJ.log((String)"");
        if (CondaDetector.isMac() || CondaDetector.isLinux()) {
            IJ.log((String)"Method 2: Parsing shell configuration files...");
            String home = System.getProperty("user.home");
            String[] configFiles = new String[]{".zshrc", ".bash_profile", ".bashrc", ".profile", ".config/fish/config.fish"};
            boolean foundInConfig = false;
            for (String configFile : configFiles) {
                Path configPath = Paths.get(home, configFile);
                if (!Files.exists(configPath, new LinkOption[0])) continue;
                IJ.log((String)("  Checking: " + configFile));
                try {
                    String content = new String(Files.readAllBytes(configPath));
                    if (content.contains("conda") || content.contains("mamba")) {
                        IJ.log((String)"    \u2192 Contains conda/mamba references");
                        String extracted = CondaDetector.extractCondaExeFromConfig(content, configPath);
                        if (extracted != null) {
                            IJ.log((String)("    \u2713 Extracted path: " + extracted));
                            foundInConfig = true;
                            continue;
                        }
                        IJ.log((String)"    \u2717 Could not extract conda path");
                        continue;
                    }
                    IJ.log((String)"    No conda references found");
                }
                catch (IOException e) {
                    IJ.log((String)"    \u2717 Could not read file");
                }
            }
            if (!foundInConfig) {
                IJ.log((String)"  No conda installation found in shell configs");
            }
            IJ.log((String)"");
        } else {
            IJ.log((String)"Method 2: Parsing shell configuration files...");
            IJ.log((String)"  Skipped (only applicable on macOS/Linux)");
            IJ.log((String)"");
        }
        IJ.log((String)"Method 3: Searching system PATH...");
        String pathResult = CondaDetector.findInPath("conda");
        if (pathResult != null) {
            IJ.log((String)("  \u2713 Found in PATH: " + pathResult));
        } else {
            IJ.log((String)"  Conda not found in PATH");
        }
        IJ.log((String)"");
        IJ.log((String)"Method 4: Checking common installation locations...");
        List<String> locations = CondaDetector.getCommonCondaLocations();
        IJ.log((String)("  Checking " + locations.size() + " potential locations:"));
        boolean foundInCommon = false;
        for (String location : locations) {
            String condaExe = CondaDetector.buildCondaExecutablePath(location);
            if (condaExe == null || !new File(condaExe).exists()) continue;
            IJ.log((String)("  \u2713 Found: " + condaExe));
            foundInCommon = true;
        }
        if (!foundInCommon) {
            IJ.log((String)"  No conda installation found in common locations");
        }
        IJ.log((String)"");
        IJ.log((String)"\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
        IJ.log((String)"");
        IJ.log((String)"Running integrated detection...");
        IJ.log((String)"");
        try {
            CondaInfo info = CondaDetector.detect();
            IJ.log((String)"");
            IJ.log((String)"\u2705 Conda detected successfully!");
            IJ.log((String)"");
            IJ.log((String)"Detection Results:");
            IJ.log((String)("  Conda Executable: " + info.getCondaExecutable()));
            IJ.log((String)("  Root Prefix:      " + info.getRootPrefix()));
            IJ.log((String)("  Version:          " + info.getVersion()));
            IJ.log((String)("  Is Micromamba:    " + info.isMicromamba()));
            IJ.log((String)"");
            IJ.log((String)"Validating installation...");
            File exeFile = new File(info.getCondaExecutable());
            IJ.log((String)("  Executable exists:   " + exeFile.exists()));
            IJ.log((String)("  Executable size:     " + exeFile.length() + " bytes"));
            IJ.log((String)("  Can execute:         " + exeFile.canExecute()));
            File rootDir = new File(info.getRootPrefix());
            IJ.log((String)("  Root prefix exists:  " + rootDir.exists()));
            IJ.log((String)("  Root prefix is dir:  " + rootDir.isDirectory()));
            IJ.log((String)"");
            IJ.log((String)"\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
            IJ.log((String)"");
            IJ.log((String)"Discovering conda environments...");
            List<CondaEnvironment> envs = CondaDetector.findAllEnvironments();
            if (envs.isEmpty()) {
                IJ.log((String)"");
                IJ.log((String)"\u26a0 No environments found");
            } else {
                IJ.log((String)"");
                IJ.log((String)("\ud83d\udce6 Found " + envs.size() + " environment(s):"));
                IJ.log((String)"");
                for (int i = 0; i < envs.size(); ++i) {
                    CondaEnvironment env = envs.get(i);
                    IJ.log((String)("  " + (i + 1) + ". " + env.getName()));
                    IJ.log((String)("     Path: " + env.getPath()));
                    File envDir = new File(env.getPath());
                    if (envDir.exists()) {
                        File envBin = new File(env.getPath(), CondaDetector.isWindows() ? "python.exe" : "bin/python");
                        if (envBin.exists()) {
                            IJ.log((String)"     Status: \u2713 Valid");
                            continue;
                        }
                        IJ.log((String)"     Status: \u26a0 Missing Python");
                        continue;
                    }
                    IJ.log((String)"     Status: \u2717 Path does not exist");
                }
                if (!envs.isEmpty()) {
                    int testIndex = new Random().nextInt(envs.size());
                    String testEnv = envs.get(testIndex).getName();
                    IJ.log((String)"");
                    IJ.log((String)"\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
                    IJ.log((String)"");
                    IJ.log((String)("Testing environment lookup for: '" + testEnv + "'"));
                    if (CondaDetector.environmentExists(testEnv)) {
                        String envPath = CondaDetector.getEnvironmentPath(testEnv);
                        IJ.log((String)"  \u2713 Environment found");
                        IJ.log((String)("  Path: " + envPath));
                    } else {
                        IJ.log((String)"  \u2717 Environment not found (unexpected)");
                    }
                }
            }
            IJ.log((String)"");
            IJ.log((String)"\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
            IJ.log((String)"");
            IJ.log((String)"\u2705 All checks completed successfully!");
        }
        catch (CondaNotFoundException e) {
            IJ.log((String)"");
            IJ.log((String)"\u274c Conda Detection Failed");
            IJ.log((String)"");
            IJ.log((String)e.getMessage());
            IJ.log((String)"");
            IJ.log((String)"\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550\u2550");
            IJ.log((String)"");
            IJ.log((String)"Troubleshooting Tips:");
            IJ.log((String)"  \u2022 Ensure conda/miniconda/anaconda is installed");
            IJ.log((String)"  \u2022 Run 'conda init' in your terminal");
            IJ.log((String)"  \u2022 Check that conda is in your PATH");
            IJ.log((String)"  \u2022 Try running 'conda --version' in terminal");
            IJ.log((String)"  \u2022 If using mambaforge, ensure it's properly installed");
        }
    }

    public static class CondaInfo {
        private final String condaExecutable;
        private final String rootPrefix;
        private final String version;

        public CondaInfo(String condaExecutable, String rootPrefix, String version) {
            this.condaExecutable = condaExecutable;
            this.rootPrefix = rootPrefix;
            this.version = version;
        }

        public String getCondaExecutable() {
            return this.condaExecutable;
        }

        public String getRootPrefix() {
            return this.rootPrefix;
        }

        public String getVersion() {
            return this.version;
        }

        public boolean isMicromamba() {
            return CondaDetector.isMicromambaExecutable(this.condaExecutable);
        }

        public String toString() {
            return String.format("CondaInfo{executable='%s', rootPrefix='%s', version='%s'}", this.condaExecutable, this.rootPrefix, this.version);
        }
    }

    public static class CondaNotFoundException
    extends Exception {
        private static final long serialVersionUID = 1L;

        public CondaNotFoundException(String message) {
            super(message);
        }
    }

    public static class CondaEnvironment {
        private final String name;
        private final String path;

        public CondaEnvironment(String name, String path) {
            this.name = name;
            this.path = path;
        }

        public String getName() {
            return this.name;
        }

        public String getPath() {
            return this.path;
        }

        public String toString() {
            return this.name + " (" + this.path + ")";
        }
    }
}

