/*
 * Decompiled with CFR 0.152.
 */
package fiji.plugin.trackmate.tracking.overlap;

import fiji.plugin.trackmate.Logger;
import fiji.plugin.trackmate.Spot;
import fiji.plugin.trackmate.SpotCollection;
import fiji.plugin.trackmate.SpotRoi;
import fiji.plugin.trackmate.tracking.SpotTracker;
import fiji.plugin.trackmate.util.Threads;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import math.geom2d.AffineTransform2D;
import math.geom2d.Point2D;
import math.geom2d.conic.Circle2D;
import math.geom2d.polygon.Polygon2D;
import math.geom2d.polygon.Polygons2D;
import math.geom2d.polygon.Rectangle2D;
import math.geom2d.polygon.SimplePolygon2D;
import net.imglib2.algorithm.MultiThreadedBenchmarkAlgorithm;
import org.jgrapht.graph.DefaultWeightedEdge;
import org.jgrapht.graph.SimpleWeightedGraph;
import org.scijava.Cancelable;

public class OverlapTracker
extends MultiThreadedBenchmarkAlgorithm
implements SpotTracker,
Cancelable {
    private SimpleWeightedGraph<Spot, DefaultWeightedEdge> graph;
    private Logger logger = Logger.VOID_LOGGER;
    private final SpotCollection spots;
    private final double enlargeFactor;
    private final IoUCalculation method;
    private final double minIoU;
    private boolean isCanceled;
    private String cancelReason;

    public OverlapTracker(SpotCollection spots, IoUCalculation method, double minIoU, double enlargeFactor) {
        this.spots = spots;
        this.method = method;
        this.minIoU = minIoU;
        this.enlargeFactor = enlargeFactor;
    }

    public SimpleWeightedGraph<Spot, DefaultWeightedEdge> getResult() {
        return this.graph;
    }

    public boolean checkInput() {
        return true;
    }

    public boolean process() {
        this.isCanceled = false;
        this.cancelReason = null;
        if (null == this.spots) {
            this.errorMessage = "[IoUTracker] The spot collection is null.";
            return false;
        }
        if (this.spots.keySet().isEmpty()) {
            this.errorMessage = "[IoUTracker] The spot collection is empty.";
            return false;
        }
        if (this.enlargeFactor <= 0.0) {
            this.errorMessage = "[IoUTracker] The enlargement factor must be strictly positive, was " + this.enlargeFactor;
            return false;
        }
        boolean empty = true;
        for (int frame : this.spots.keySet()) {
            if (this.spots.getNSpots(frame, true) <= 0) continue;
            empty = false;
            break;
        }
        if (empty) {
            this.errorMessage = "[IoUTracker] The spot collection is empty.";
            return false;
        }
        long start = System.currentTimeMillis();
        this.graph = new SimpleWeightedGraph(DefaultWeightedEdge.class);
        AtomicBoolean ok = new AtomicBoolean(true);
        Iterator<Integer> frameIterator = this.spots.keySet().iterator();
        int sourceFrame = frameIterator.next();
        Map<Spot, Polygon2D> sourceGeometries = OverlapTracker.createGeometry(this.spots.iterable(sourceFrame, true), this.method, this.enlargeFactor);
        this.logger.setStatus("Frame to frame linking...");
        int progress = 0;
        while (frameIterator.hasNext() && ok.get() && !this.isCanceled()) {
            int targetFrame = frameIterator.next();
            Map<Spot, Polygon2D> targetGeometries = OverlapTracker.createGeometry(this.spots.iterable(targetFrame, true), this.method, this.enlargeFactor);
            if (sourceGeometries.isEmpty() || targetGeometries.isEmpty()) continue;
            ExecutorService executors = Threads.newFixedThreadPool(this.numThreads);
            ArrayList<Future<IoULink>> futures = new ArrayList<Future<IoULink>>();
            for (Spot spot : targetGeometries.keySet()) {
                Polygon2D targetPoly = targetGeometries.get(spot);
                futures.add(executors.submit(new FindBestSourceTask(spot, targetPoly, sourceGeometries, this.minIoU)));
            }
            for (Future future : futures) {
                if (!ok.get() || this.isCanceled()) break;
                try {
                    IoULink link = (IoULink)future.get();
                    if (link.source == null) continue;
                    this.graph.addVertex((Object)link.source);
                    this.graph.addVertex((Object)link.target);
                    DefaultWeightedEdge edge = (DefaultWeightedEdge)this.graph.addEdge((Object)link.source, (Object)link.target);
                    this.graph.setEdgeWeight((Object)edge, 1.0 - link.iou);
                }
                catch (InterruptedException | ExecutionException e) {
                    this.errorMessage = e.getMessage();
                    ok.set(false);
                }
            }
            executors.shutdown();
            sourceGeometries = targetGeometries;
            this.logger.setProgress((double)progress++ / (double)this.spots.keySet().size());
        }
        this.logger.setProgress(1.0);
        this.logger.setStatus("");
        long end = System.currentTimeMillis();
        this.processingTime = end - start;
        return ok.get();
    }

    @Override
    public void setLogger(Logger logger) {
        this.logger = logger;
    }

    protected boolean checkSettingsValidity(Map<String, Object> settings, StringBuilder str) {
        if (null == settings) {
            str.append("Settings map is null.\n");
            return false;
        }
        boolean ok = true;
        return true;
    }

    private static Map<Spot, Polygon2D> createGeometry(Iterable<Spot> spots, IoUCalculation method, double scale) {
        HashMap<Spot, Object> geometries = new HashMap<Spot, Object>();
        switch (method.ordinal()) {
            case 0: {
                for (Spot spot : spots) {
                    geometries.put(spot, OverlapTracker.toBoundingBox(spot, scale));
                }
                break;
            }
            case 1: {
                for (Spot spot : spots) {
                    geometries.put(spot, OverlapTracker.toPolygon(spot, scale));
                }
                break;
            }
            default: {
                throw new IllegalArgumentException("Do not know how to compute IoU for method " + String.valueOf((Object)method));
            }
        }
        return Collections.unmodifiableMap(geometries);
    }

    private static SimplePolygon2D toPolygon(Spot spot, double scale) {
        SimplePolygon2D poly;
        double xc = spot.getDoublePosition(0);
        double yc = spot.getDoublePosition(1);
        SpotRoi roi = spot.getRoi();
        if (roi == null) {
            double radius = spot.getFeature("RADIUS");
            poly = new SimplePolygon2D(new Circle2D(xc, yc, radius).asPolyline(32));
        } else {
            double[] xcoords = roi.toPolygonX(1.0, 0.0, xc, 1.0);
            double[] ycoords = roi.toPolygonY(1.0, 0.0, yc, 1.0);
            poly = new SimplePolygon2D(xcoords, ycoords);
        }
        return poly.transform(AffineTransform2D.createScaling((Point2D)new Point2D(xc, yc), (double)scale, (double)scale));
    }

    private static Rectangle2D toBoundingBox(Spot spot, double scale) {
        double xc = spot.getDoublePosition(0);
        double yc = spot.getDoublePosition(1);
        SpotRoi roi = spot.getRoi();
        if (roi == null) {
            double radius = spot.getFeature("RADIUS") * scale;
            return new Rectangle2D(xc - radius, yc - radius, 2.0 * radius, 2.0 * radius);
        }
        double minX = Arrays.stream(roi.x).min().getAsDouble() * scale;
        double maxX = Arrays.stream(roi.x).max().getAsDouble() * scale;
        double minY = Arrays.stream(roi.y).min().getAsDouble() * scale;
        double maxY = Arrays.stream(roi.y).max().getAsDouble() * scale;
        return new Rectangle2D(xc + minX, yc + minY, maxX - minX, maxY - minY);
    }

    public boolean isCanceled() {
        return this.isCanceled;
    }

    public void cancel(String reason) {
        this.isCanceled = true;
        this.cancelReason = reason;
    }

    public String getCancelReason() {
        return this.cancelReason;
    }

    public static enum IoUCalculation {
        FAST("Fast", "IoU is calculated using the bounding box of the spot."),
        PRECISE("Precise", "IoU is calculated over the shape of the spot ROI.");

        private final String str;
        private final String infoText;

        private IoUCalculation(String str, String infoText) {
            this.str = str;
            this.infoText = infoText;
        }

        public String getInfoText() {
            return this.infoText;
        }

        public String toString() {
            return this.str;
        }
    }

    private static final class FindBestSourceTask
    implements Callable<IoULink> {
        private final Spot target;
        private final Polygon2D targetPoly;
        private final Map<Spot, Polygon2D> sourceGeometries;
        private final double minIoU;

        public FindBestSourceTask(Spot target, Polygon2D targetPoly, Map<Spot, Polygon2D> sourceGeometries, double minIoU) {
            this.target = target;
            this.targetPoly = targetPoly;
            this.sourceGeometries = sourceGeometries;
            this.minIoU = minIoU;
        }

        @Override
        public IoULink call() throws Exception {
            double targetArea = Math.abs(this.targetPoly.area());
            double maxIoU = this.minIoU;
            Spot bestSpot = null;
            for (Spot spot : this.sourceGeometries.keySet()) {
                double union;
                double iou;
                Polygon2D sourcePoly = this.sourceGeometries.get(spot);
                double intersection = Math.abs(Polygons2D.intersection((Polygon2D)this.targetPoly, (Polygon2D)sourcePoly).area());
                if (intersection == 0.0 || !((iou = intersection / (union = Math.abs(sourcePoly.area()) + targetArea - intersection)) > maxIoU)) continue;
                maxIoU = iou;
                bestSpot = spot;
            }
            return new IoULink(bestSpot, this.target, maxIoU);
        }
    }

    private static final class IoULink {
        public final Spot source;
        public final Spot target;
        public final double iou;

        public IoULink(Spot source, Spot target, double iou) {
            this.source = source;
            this.target = target;
            this.iou = iou;
        }
    }
}

