/*
 * Decompiled with CFR 0.152.
 */
package fiji.plugin.trackmate.action;

import fiji.plugin.trackmate.Logger;
import fiji.plugin.trackmate.Model;
import fiji.plugin.trackmate.SelectionModel;
import fiji.plugin.trackmate.Settings;
import fiji.plugin.trackmate.Spot;
import fiji.plugin.trackmate.TrackMate;
import fiji.plugin.trackmate.action.AbstractTMAction;
import fiji.plugin.trackmate.action.TrackMateAction;
import fiji.plugin.trackmate.action.TrackMateActionFactory;
import fiji.plugin.trackmate.gui.Icons;
import fiji.plugin.trackmate.gui.displaysettings.DisplaySettings;
import fiji.plugin.trackmate.util.TMUtils;
import fiji.plugin.trackmate.visualization.trackscheme.SpotIconGrabber;
import ij.CompositeImage;
import ij.ImagePlus;
import ij.ImageStack;
import ij.gui.GenericDialog;
import ij.measure.Calibration;
import ij.process.ImageProcessor;
import java.awt.Frame;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import javax.swing.ImageIcon;
import net.imagej.ImgPlus;
import net.imglib2.RandomAccessibleInterval;
import net.imglib2.img.Img;
import net.imglib2.img.display.imagej.ImageJFunctions;
import net.imglib2.view.Views;
import org.jgrapht.graph.DefaultWeightedEdge;
import org.scijava.plugin.Plugin;

public class ExtractTrackStackAction
extends AbstractTMAction {
    public static final String NAME = "Extract track stack";
    public static final String KEY = "EXTRACT_TRACK_STACK";
    public static final String INFO_TEXT = "<html> Generate a stack of images taken from the track that joins two selected spots. <p> There must be exactly 1 or 2 spots selected for this action to work. If only one spot is selected, then the stack is extracted from the track it belongs to, from the first spot in time to the last in time. If there are two spots selected, they must belong to a track that connects them. A path is then found that joins them and the stack is extracted from this path.<p> A stack of images will be generated from the spots that join them. A GUI allows specifying the size of the extract, in units of the largest spot in the track, and whether to capture a 2D or 3D stack over time. All channels are captured. </html>";
    private static double diameterFactor = 1.5;
    private static int dimChoice = 0;
    private static final float RESIZE_FACTOR = 1.5f;

    /*
     * Enabled aggressive block sorting
     */
    @Override
    public void execute(TrackMate trackmate, SelectionModel selectionModel, DisplaySettings displaySettings, Frame parent) {
        Spot start1;
        Spot end1;
        GenericDialog dialog = new GenericDialog(NAME, parent);
        dialog.addSlider("Image size (spot\ndiameter units):", 0.1, 5.1, diameterFactor);
        String[] dimChoices = new String[]{"Central slice ", "3D"};
        dialog.addRadioButtonGroup("Dimensionality:", dimChoices, 2, 1, dimChoices[dimChoice]);
        dialog.showDialog();
        if (dialog.wasCanceled()) {
            return;
        }
        diameterFactor = dialog.getNextNumber();
        dimChoice = Arrays.asList(dimChoices).indexOf(dialog.getNextRadioButton());
        boolean do3d = dimChoice == 1;
        this.logger.log("Capturing " + (do3d ? "3D" : "2D") + " track stack.\n");
        Model model = trackmate.getModel();
        Set<Spot> selection = selectionModel.getSpotSelection();
        int nspots = selection.size();
        if (nspots != 2) {
            if (nspots != 1) {
                this.logger.error("Expected 1 or 2 spots in the selection, got " + nspots + ".\nAborting.\n");
                return;
            }
            Spot spot = selection.iterator().next();
            Integer trackID = model.getTrackModel().trackIDOf(spot);
            ArrayList<Spot> spots = new ArrayList<Spot>(model.getTrackModel().trackSpots(trackID));
            Collections.sort(spots, Spot.frameComparator);
            Spot start = (Spot)spots.get(0);
            Spot end = (Spot)spots.get(spots.size() - 1);
            selectionModel.clearSelection();
            selectionModel.addSpotToSelection(start);
            selectionModel.addSpotToSelection(end);
            List<DefaultWeightedEdge> edges = model.getTrackModel().dijkstraShortestPath(start, end);
            if (null == edges) {
                this.logger.error("The 2 spots are not connected.\nAborting\n");
                return;
            }
            selectionModel.addEdgeToSelection(edges);
            ImagePlus imp = ExtractTrackStackAction.trackStack(trackmate, spot, do3d, this.logger);
            imp.show();
            imp.setZ(imp.getNSlices() / 2 + 1);
            imp.resetDisplayRange();
            return;
        }
        Iterator<Spot> it = selection.iterator();
        Spot start = it.next();
        Spot end = it.next();
        selectionModel.clearSelection();
        selectionModel.addSpotToSelection(start);
        selectionModel.addSpotToSelection(end);
        if (start.getFeature("POSITION_T") > end.getFeature("POSITION_T")) {
            end1 = start;
            start1 = end;
        } else {
            end1 = end;
            start1 = start;
        }
        List<DefaultWeightedEdge> edges = model.getTrackModel().dijkstraShortestPath(start1, end1);
        if (null == edges) {
            this.logger.error("The 2 spots are not connected.\nAborting\n");
            return;
        }
        selectionModel.addEdgeToSelection(edges);
        ImagePlus imp = ExtractTrackStackAction.trackStack(trackmate, start1, end1, do3d, this.logger);
        imp.show();
        imp.setZ(imp.getNSlices() / 2 + 1);
        imp.resetDisplayRange();
    }

    public static final ImagePlus trackStack(TrackMate trackmate, Spot spot, boolean do3d, Logger logger) {
        Model model = trackmate.getModel();
        Integer trackID = model.getTrackModel().trackIDOf(spot);
        ArrayList<Spot> spots = new ArrayList<Spot>(model.getTrackModel().trackSpots(trackID));
        Collections.sort(spots, Spot.frameComparator);
        Spot start = (Spot)spots.get(0);
        Spot end = (Spot)spots.get(spots.size() - 1);
        return ExtractTrackStackAction.trackStack(trackmate, start, end, do3d, logger);
    }

    public static final ImagePlus trackStack(TrackMate trackmate, Spot start, Spot end, boolean do3d, Logger logger) {
        Spot start1;
        Spot end1;
        Model model = trackmate.getModel();
        if (start.getFeature("POSITION_T") > end.getFeature("POSITION_T")) {
            end1 = start;
            start1 = end;
        } else {
            end1 = end;
            start1 = start;
        }
        List<DefaultWeightedEdge> edges = model.getTrackModel().dijkstraShortestPath(start1, end1);
        if (null == edges) {
            logger.error("The 2 spots are not connected.\nAborting\n");
            return null;
        }
        ArrayList<Spot> path = new ArrayList<Spot>(edges.size());
        path.add(start1);
        Spot previous = start1;
        double radius = Math.abs(start1.getFeature("RADIUS")) * diameterFactor;
        for (DefaultWeightedEdge edge : edges) {
            Spot current = model.getTrackModel().getEdgeSource(edge);
            if (current == previous) {
                current = model.getTrackModel().getEdgeTarget(edge);
            }
            path.add(current);
            double ct = Math.abs(current.getFeature("RADIUS"));
            if (ct > radius) {
                radius = ct;
            }
            previous = current;
        }
        path.add(end1);
        TreeSet<Spot> sortedSpots = new TreeSet<Spot>(Spot.timeComparator);
        sortedSpots.addAll(path);
        return ExtractTrackStackAction.trackStack(trackmate.getSettings(), path, radius, do3d, logger);
    }

    public static final ImagePlus trackStack(Settings settings, List<Spot> path, double radius, boolean do3d, Logger logger) {
        int nspots = path.size();
        double[] calibration = TMUtils.getSpatialCalibration(settings.imp);
        int width = (int)Math.ceil(2.0 * radius * 1.5 / calibration[0]);
        int height = (int)Math.ceil(2.0 * radius * 1.5 / calibration[1]);
        int depth = do3d ? (int)Math.ceil(2.0 * radius * 1.5 / calibration[2]) : 1;
        ImgPlus img = TMUtils.rawWraps(settings.imp);
        ImageStack stack = new ImageStack(width, height);
        int progress = 0;
        int nChannels = settings.imp.getNChannels();
        for (Spot spot : path) {
            int frame = spot.getFeature("FRAME").intValue();
            for (int c = 0; c < nChannels; ++c) {
                Img crop;
                ImgPlus imgCT = TMUtils.hyperSlice(img, c, frame);
                int x = (int)(Math.round(spot.getFeature("POSITION_X") / calibration[0]) - (long)(width / 2));
                int y = (int)(Math.round(spot.getFeature("POSITION_Y") / calibration[1]) - (long)(height / 2));
                long slice = 0L;
                if (imgCT.numDimensions() > 2) {
                    slice = Math.round(spot.getFeature("POSITION_Z") / calibration[2]);
                    if (slice < 0L) {
                        slice = 0L;
                    }
                    if (slice >= imgCT.dimension(2)) {
                        slice = imgCT.dimension(2) - 1L;
                    }
                }
                SpotIconGrabber grabber = new SpotIconGrabber(imgCT);
                if (do3d) {
                    crop = grabber.grabImage(x, y, slice, width, height, depth);
                    int i = 0;
                    while ((long)i < crop.dimension(2)) {
                        ImageProcessor processor = ImageJFunctions.wrap((RandomAccessibleInterval)Views.hyperSlice(crop, (int)2, (long)i), (String)crop.toString()).getProcessor();
                        stack.addSlice(spot.toString(), processor);
                        ++i;
                    }
                    continue;
                }
                crop = grabber.grabImage(x, y, slice, width, height);
                stack.addSlice(spot.toString(), ImageJFunctions.wrap(crop, (String)crop.toString()).getProcessor());
            }
            logger.setProgress((float)(progress + 1) / (float)nspots);
            ++progress;
        }
        ImagePlus stackTrack = new ImagePlus("", stack);
        stackTrack.setTitle("Path from " + String.valueOf(path.get(0)) + " to " + String.valueOf(path.get(path.size() - 1)));
        Calibration impCal = stackTrack.getCalibration();
        impCal.setTimeUnit(settings.imp.getCalibration().getTimeUnit());
        impCal.setUnit(settings.imp.getCalibration().getUnit());
        impCal.pixelWidth = calibration[0];
        impCal.pixelHeight = calibration[1];
        impCal.pixelDepth = calibration[2];
        impCal.frameInterval = settings.dt;
        stackTrack.setDimensions(nChannels, depth, nspots);
        stackTrack.setOpenAsHyperStack(true);
        logger.log("Done.");
        if (nChannels > 1) {
            CompositeImage cmp = new CompositeImage(stackTrack, 1);
            if (settings.imp instanceof CompositeImage) {
                CompositeImage scmp = (CompositeImage)settings.imp;
                for (int c = 0; c < nChannels; ++c) {
                    cmp.setChannelLut(scmp.getChannelLut(c + 1), c + 1);
                }
            }
            return cmp;
        }
        return stackTrack;
    }

    @Plugin(type=TrackMateActionFactory.class)
    public static class ExtractTrackStackActionFactory
    implements TrackMateActionFactory {
        @Override
        public String getInfoText() {
            return ExtractTrackStackAction.INFO_TEXT;
        }

        @Override
        public String getName() {
            return ExtractTrackStackAction.NAME;
        }

        @Override
        public String getKey() {
            return ExtractTrackStackAction.KEY;
        }

        @Override
        public ImageIcon getIcon() {
            return Icons.MAGNIFIER_ICON;
        }

        @Override
        public TrackMateAction create() {
            return new ExtractTrackStackAction();
        }
    }
}

