/*
 * Decompiled with CFR 0.152.
 */
package io.scif.formats;

import io.scif.AbstractChecker;
import io.scif.AbstractFormat;
import io.scif.AbstractMetadata;
import io.scif.AbstractParser;
import io.scif.AbstractTranslator;
import io.scif.AbstractWriter;
import io.scif.ByteArrayPlane;
import io.scif.ByteArrayReader;
import io.scif.Format;
import io.scif.FormatException;
import io.scif.HasColorTable;
import io.scif.ImageMetadata;
import io.scif.MetaTable;
import io.scif.Plane;
import io.scif.UnsupportedCompressionException;
import io.scif.codec.BitBuffer;
import io.scif.codec.CodecOptions;
import io.scif.codec.CodecService;
import io.scif.codec.JPEGCodec;
import io.scif.codec.MSRLECodec;
import io.scif.codec.MSVideoCodec;
import io.scif.config.SCIFIOConfig;
import io.scif.util.FormatTools;
import io.scif.util.ImageTools;
import io.scif.util.SCIFIOMetadataTools;
import java.awt.image.IndexColorModel;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import net.imagej.axis.Axes;
import net.imagej.axis.AxisType;
import net.imglib2.FinalInterval;
import net.imglib2.Interval;
import net.imglib2.display.ColorTable;
import net.imglib2.display.ColorTable8;
import org.scijava.io.handle.DataHandle;
import org.scijava.io.handle.DataHandleService;
import org.scijava.io.location.Location;
import org.scijava.plugin.Parameter;
import org.scijava.plugin.Plugin;

@Plugin(type=Format.class, name="Audio Video Interleave")
public class AVIFormat
extends AbstractFormat {
    private static final int MSRLE = 1;
    private static final int MS_VIDEO = 1296126531;
    private static final int JPEG = 1196444237;
    private static final int Y8 = 538982489;
    private static final byte[] MJPEG_HUFFMAN_TABLE = new byte[]{-1, -60, 1, -94, 0, 0, 1, 5, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 1, 0, 3, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 16, 0, 2, 1, 3, 3, 2, 4, 3, 5, 5, 4, 4, 0, 0, 1, 125, 1, 2, 3, 0, 4, 17, 5, 18, 33, 49, 65, 6, 19, 81, 97, 7, 34, 113, 20, 50, -127, -111, -95, 8, 35, 66, -79, -63, 21, 82, -47, -16, 36, 51, 98, 114, -126, 9, 10, 22, 23, 24, 25, 26, 37, 38, 39, 40, 41, 42, 52, 53, 54, 55, 56, 57, 58, 67, 68, 69, 70, 71, 72, 73, 74, 83, 84, 85, 86, 87, 88, 89, 90, 99, 100, 101, 102, 103, 104, 105, 106, 115, 116, 117, 118, 119, 120, 121, 122, -125, -124, -123, -122, -121, -120, -119, -118, -110, -109, -108, -107, -106, -105, -104, -103, -102, -94, -93, -92, -91, -90, -89, -88, -87, -86, -78, -77, -76, -75, -74, -73, -72, -71, -70, -62, -61, -60, -59, -58, -57, -56, -55, -54, -46, -45, -44, -43, -42, -41, -40, -39, -38, -31, -30, -29, -28, -27, -26, -25, -24, -23, -22, -15, -14, -13, -12, -11, -10, -9, -8, -7, -6, 17, 0, 2, 1, 2, 4, 4, 3, 4, 7, 5, 4, 4, 0, 1, 2, 119, 0, 1, 2, 3, 17, 4, 5, 33, 49, 6, 18, 65, 81, 7, 97, 113, 19, 34, 50, -127, 8, 20, 66, -111, -95, -79, -63, 9, 35, 51, 82, -16, 21, 98, 114, -47, 10, 22, 36, 52, -31, 37, -15, 23, 24, 25, 26, 38, 39, 40, 41, 42, 53, 54, 55, 56, 57, 58, 67, 68, 69, 70, 71, 72, 73, 74, 83, 84, 85, 86, 87, 88, 89, 90, 99, 100, 101, 102, 103, 104, 105, 106, 115, 116, 117, 118, 119, 120, 121, 122, -126, -125, -124, -123, -122, -121, -120, -119, -118, -110, -109, -108, -107, -106, -105, -104, -103, -102, -94, -93, -92, -91, -90, -89, -88, -87, -86, -78, -77, -76, -75, -74, -73, -72, -71, -70, -62, -61, -60, -59, -58, -57, -56, -55, -54, -46, -45, -44, -43, -42, -41, -40, -39, -38, -30, -29, -28, -27, -26, -25, -24, -23, -22, -14, -13, -12, -11, -10, -9, -8, -7, -6};

    @Override
    protected String[] makeSuffixArray() {
        return new String[]{"avi"};
    }

    private static class AVIUtils {
        private AVIUtils() {
        }

        public static CodecOptions createCodecOptions(Metadata meta, int imageIndex, long planeIndex) {
            CodecOptions options = new CodecOptions();
            options.width = (int)meta.get(imageIndex).getAxisLength(Axes.X);
            options.height = (int)meta.get(imageIndex).getAxisLength(Axes.Y);
            options.previousImage = meta.getLastPlaneIndex() == planeIndex - 1L ? meta.getLastPlaneBytes() : null;
            options.bitsPerSample = meta.getBmpBitsPerPixel();
            options.interleaved = meta.get(imageIndex).getInterleavedAxisCount() > 0;
            options.littleEndian = meta.get(imageIndex).isLittleEndian();
            return options;
        }

        private static String getCodecName(int bmpCompression) {
            switch (bmpCompression) {
                case 0: {
                    return "Raw (uncompressed)";
                }
                case 1: {
                    return "Microsoft Run-Length Encoding (MSRLE)";
                }
                case 1296126531: {
                    return "Microsoft Video (MSV1)";
                }
                case 1196444237: {
                    return "JPEG";
                }
            }
            return "Unknown";
        }

        public static byte[] extractCompression(Metadata meta, CodecOptions options, DataHandle<Location> stream, ByteArrayPlane plane, long planeIndex, int[] dims) throws IOException, FormatException {
            int bmpCompression = meta.getBmpCompression();
            long fileOff = meta.getOffsets().get((int)planeIndex);
            long filePointer = stream.offset();
            stream.seek(fileOff);
            byte[] buf = null;
            CodecService codecService = (CodecService)meta.context().service(CodecService.class);
            if (bmpCompression == 1) {
                byte[] b = new byte[(int)meta.getLengths().get((int)planeIndex).longValue()];
                stream.read(b);
                MSRLECodec codec = codecService.getCodec(MSRLECodec.class);
                buf = codec.decompress(b, options);
                plane.setData(buf);
                if (AVIUtils.updateLastPlane(meta, planeIndex, dims)) {
                    meta.setLastPlane(plane);
                    meta.setLastPlaneIndex(planeIndex);
                    meta.setLastDimensions(dims);
                }
            } else if (bmpCompression == 1296126531) {
                MSVideoCodec codec = codecService.getCodec(MSVideoCodec.class);
                buf = codec.decompress(stream, options);
                plane.setData(buf);
                if (AVIUtils.updateLastPlane(meta, planeIndex, dims)) {
                    meta.setLastPlane(plane);
                    meta.setLastPlaneIndex(planeIndex);
                    meta.setLastDimensions(dims);
                }
            } else if (bmpCompression == 1196444237) {
                JPEGCodec codec = codecService.getCodec(JPEGCodec.class);
                byte[] tmpPlane = new byte[(int)meta.getLengths().get((int)planeIndex).longValue()];
                stream.read(tmpPlane);
                boolean motionJPEG = new String(tmpPlane, 6, 4, "UTF-8").equals("AVI1");
                if (motionJPEG) {
                    byte[] fixedPlane = new byte[tmpPlane.length + MJPEG_HUFFMAN_TABLE.length];
                    System.arraycopy(plane.getBytes(), 0, fixedPlane, 0, 20);
                    System.arraycopy(MJPEG_HUFFMAN_TABLE, 0, fixedPlane, 20, MJPEG_HUFFMAN_TABLE.length);
                    System.arraycopy(plane.getBytes(), 20, fixedPlane, 20 + MJPEG_HUFFMAN_TABLE.length, tmpPlane.length - 20);
                    tmpPlane = fixedPlane;
                }
                buf = codec.decompress(tmpPlane, options);
                if (motionJPEG) {
                    buf = plane.getBytes();
                    for (int i = 0; i < buf.length; i += 3) {
                        int y = buf[i] & 0xFF;
                        int cb = (buf[i + 1] & 0xFF) - 128;
                        int cr = (buf[i + 2] & 0xFF) - 128;
                        int red = (int)((double)y + 1.402 * (double)cr);
                        int green = (int)((double)y - 0.34414 * (double)cb - 0.71414 * (double)cr);
                        int blue = (int)((double)y + 1.772 * (double)cb);
                        if (red < 0) {
                            red = 0;
                        } else if (red > 255) {
                            red = 255;
                        }
                        if (green < 0) {
                            green = 0;
                        } else if (green > 255) {
                            green = 255;
                        }
                        if (blue < 0) {
                            blue = 0;
                        } else if (blue > 255) {
                            blue = 255;
                        }
                        buf[i] = (byte)(red & 0xFF);
                        buf[i + 1] = (byte)(green & 0xFF);
                        buf[i + 2] = (byte)(blue & 0xFF);
                    }
                }
            } else {
                throw new UnsupportedCompressionException(bmpCompression + " not supported");
            }
            stream.seek(filePointer);
            return buf;
        }

        private static boolean updateLastPlane(Metadata meta, long planeIndex, int[] dims) {
            if (meta.getLastPlaneIndex() != planeIndex) {
                return true;
            }
            int[] lastDims = meta.getLastDimensions();
            boolean smaller = dims[0] > lastDims[0] && dims[1] > lastDims[1] && dims[2] < lastDims[2] && dims[3] < lastDims[3];
            return !smaller;
        }
    }

    @Plugin(type=io.scif.Translator.class, priority=-100.0)
    public static class Translator
    extends AbstractTranslator<io.scif.Metadata, Metadata> {
        @Override
        public Class<? extends io.scif.Metadata> source() {
            return io.scif.Metadata.class;
        }

        @Override
        public Class<? extends io.scif.Metadata> dest() {
            return Metadata.class;
        }

        @Override
        protected void translateImageMetadata(List<ImageMetadata> source, Metadata dest) {
            ArrayList<Long> offsets = new ArrayList<Long>();
            ArrayList<Long> lengths = new ArrayList<Long>();
            dest.setOffsets(offsets);
            dest.setLengths(lengths);
            dest.add(source.get(0).copy());
            int sizeX = (int)source.get(0).getAxisLength(Axes.X);
            int sizeY = (int)source.get(0).getAxisLength(Axes.Y);
            int bpp = source.get(0).getBitsPerPixel();
            long length = bpp;
            for (long l : source.get(0).getAxesLengthsPlanar()) {
                length *= l;
            }
            long offset = 0L;
            int npad = sizeX % 4;
            if (npad != 0) {
                sizeX += 4 - npad;
            }
            int numChannels = (int)source.get(0).getAxisLength(Axes.CHANNEL);
            dest.setBmpBitsPerPixel((short)(bpp * numChannels));
            dest.setBmpWidth(sizeX * (bpp / 8));
            dest.setBmpScanLineSize(dest.getBmpWidth() * numChannels);
            try {
                offset = dest.getSource() == null ? 0L : dest.getSource().offset();
            }
            catch (IOException e) {
                this.log().error((Object)"Error retrieving AVI plane offset", (Throwable)e);
            }
            ImageMetadata destImageMeta = dest.get(0);
            long nonplanarChannels = 1L;
            if (!destImageMeta.isMultichannel()) {
                nonplanarChannels = numChannels;
                length *= (long)numChannels;
            }
            int i = 0;
            while ((long)i < source.get(0).getPlaneCount() / nonplanarChannels) {
                offsets.add(offset);
                lengths.add(length);
                offset += length;
                ++i;
            }
            if (numChannels > 1) {
                dest.setBmpColorsUsed((int)Math.pow(2.0, bpp));
            }
            dest.setBmpCompression(0);
            if (HasColorTable.class.isAssignableFrom(source.getClass())) {
                ColorTable ct = ((HasColorTable)((Object)source)).getColorTable(0, 0L);
                dest.setBmpColorsUsed(ct.getLength());
            }
            dest.setBytesPerPlane((int)length / 8);
        }
    }

    public static class Writer
    extends AbstractWriter<Metadata> {
        private static final long SAVE_MOVI = 4092L;
        private static final long SAVE_FILE_SIZE = 4L;
        private static final long SAVE_STRF_SIZE = 168L;
        private static final long SAVE_STRN_POS = 1236L;
        private static final long SAVE_JUNK_SIG = 1260L;
        private static final long SAVE_LIST1_SIZE = 16L;
        private static final long SAVE_LIST1_SUBSIZE = 92L;
        private static final long FRAME_OFFSET = 48L;
        private static final long FRAME_OFFSET_2 = 140L;
        private static final long PADDING_BYTES = 2816L;
        private static final long SAVE_LIST2_SIZE = 4088L;
        private static final String DATA_SIGNATURE = "00db";
        private int planesWritten = 0;
        private int bytesPerPixel;
        private int xDim;
        private int yDim;
        private int zDim;
        private int tDim;
        private int xPad;
        private int microSecPerFrame;
        private List<Long> savedbLength;
        private long idx1Pos;
        private long endPos;
        private long saveidx1Length;
        @Parameter
        private DataHandleService dataHandleService;

        @Override
        protected String[] makeCompressionTypes() {
            return new String[0];
        }

        @Override
        public void writePlane(int imageIndex, long planeIndex, Plane plane, Interval bounds) throws FormatException, IOException {
            Metadata meta = (Metadata)this.getMetadata();
            byte[] buf = plane.getBytes();
            boolean interleaved = plane.getImageMetadata().getInterleavedAxisCount() > 0;
            this.checkParams(imageIndex, planeIndex, buf, bounds);
            if (!SCIFIOMetadataTools.wholePlane(imageIndex, meta, bounds)) {
                throw new FormatException("AVIWriter does not yet support saving image tiles.");
            }
            int nChannels = (int)plane.getImageMetadata().getAxisLength(Axes.CHANNEL);
            int width = this.xDim - this.xPad;
            int height = buf.length / (width * this.bytesPerPixel);
            DataHandle<Location> handle = this.getHandle();
            handle.seek(this.idx1Pos);
            handle.writeBytes(DATA_SIGNATURE);
            this.savedbLength.add(handle.offset());
            handle.writeInt(this.bytesPerPixel * this.xDim * this.yDim);
            int rowPad = this.xPad * this.bytesPerPixel;
            byte[] rowBuffer = new byte[width * this.bytesPerPixel + rowPad];
            for (int row = height - 1; row >= 0; --row) {
                for (int col = 0; col < width; ++col) {
                    int offset = row * width + col;
                    if (interleaved) {
                        offset *= nChannels;
                    }
                    byte r = buf[offset];
                    if (nChannels > 1) {
                        byte g = buf[offset + (interleaved ? 1 : width * height)];
                        byte b = 0;
                        if (nChannels > 2) {
                            b = buf[offset + (interleaved ? 2 : 2 * width * height)];
                        }
                        rowBuffer[col * this.bytesPerPixel] = b;
                        rowBuffer[col * this.bytesPerPixel + 1] = g;
                    }
                    rowBuffer[col * this.bytesPerPixel + this.bytesPerPixel - 1] = r;
                }
                handle.write(rowBuffer);
            }
            ++this.planesWritten;
            this.idx1Pos = handle.offset();
            handle.seek(4088L);
            handle.writeInt((int)(this.idx1Pos - 4092L));
            handle.seek(this.idx1Pos);
            handle.writeBytes("idx1");
            this.saveidx1Length = handle.offset();
            handle.writeInt(4 + this.planesWritten * 16);
            for (int z = 0; z < this.planesWritten; ++z) {
                handle.writeBytes(DATA_SIGNATURE);
                if (z == 0) {
                    handle.writeInt(16);
                } else {
                    handle.writeInt(0);
                }
                handle.writeInt((int)(this.savedbLength.get(z) - 4L - 4092L));
                handle.writeInt(this.bytesPerPixel * this.xDim * this.yDim);
            }
            this.endPos = handle.offset();
            handle.seek(4L);
            handle.writeInt((int)(this.endPos - 8L));
            handle.seek(this.saveidx1Length);
            handle.writeInt((int)(this.endPos - (this.saveidx1Length + 4L)));
            handle.seek(48L);
            handle.writeInt(this.planesWritten);
            handle.seek(140L);
            handle.writeInt(this.planesWritten);
        }

        @Override
        public boolean canDoStacks() {
            return true;
        }

        @Override
        public int[] getPixelTypes(String codec) {
            return new int[]{1};
        }

        @Override
        public void close() throws IOException {
            super.close();
            this.planesWritten = 0;
            this.bytesPerPixel = 0;
            this.xPad = 0;
            this.tDim = 0;
            this.zDim = 0;
            this.yDim = 0;
            this.xDim = 0;
            this.microSecPerFrame = 0;
            this.savedbLength = null;
            this.idx1Pos = 0L;
            this.endPos = 0L;
            this.saveidx1Length = 0L;
        }

        @Override
        public void setDest(DataHandle<Location> out, int imageIndex, SCIFIOConfig config) throws FormatException, IOException {
            super.setDest(out, imageIndex, config);
            this.savedbLength = new ArrayList<Long>();
            Metadata meta = (Metadata)this.getMetadata();
            if (out.length() > 0L) {
                DataHandle in = (DataHandle)this.dataHandleService.create(out.get());
                in.setOrder(DataHandle.ByteOrder.LITTLE_ENDIAN);
                in.seek(48L);
                this.planesWritten = in.readInt();
                in.seek(4L);
                this.endPos = (long)in.readInt() + 4L + 4L;
                in.seek(4088L);
                this.idx1Pos = (long)in.readInt() + 4088L + 4L;
                this.saveidx1Length = this.idx1Pos + 4L;
                if (this.planesWritten > 0) {
                    in.seek(this.saveidx1Length + 4L);
                }
                for (int z = 0; z < this.planesWritten; ++z) {
                    in.skipBytes(8);
                    this.savedbLength.add((long)(in.readInt() + 4) + 4092L);
                    in.skipBytes(4);
                }
                in.close();
                out.seek(this.idx1Pos);
            }
            out.setOrder(DataHandle.ByteOrder.LITTLE_ENDIAN);
            ImageMetadata imageMetadata = meta.get(imageIndex);
            this.tDim = (int)imageMetadata.getAxisLength(Axes.Z);
            this.zDim = (int)imageMetadata.getAxisLength(Axes.TIME);
            this.yDim = (int)imageMetadata.getAxisLength(Axes.Y);
            this.xDim = (int)imageMetadata.getAxisLength(Axes.X);
            String type = FormatTools.getPixelTypeString(imageMetadata.getPixelType());
            int pixelType = FormatTools.pixelTypeFromString(type);
            this.bytesPerPixel = FormatTools.getBytesPerPixel(pixelType);
            this.bytesPerPixel = (int)((long)this.bytesPerPixel * imageMetadata.getAxisLength(Axes.CHANNEL));
            this.xPad = 0;
            int xMod = this.xDim % 4;
            if (xMod != 0) {
                this.xPad = 4 - xMod;
                this.xDim += this.xPad;
            }
            byte[][] lut = null;
            if (this.getColorModel() instanceof IndexColorModel) {
                lut = new byte[4][256];
                IndexColorModel model = (IndexColorModel)this.getColorModel();
                model.getReds(lut[0]);
                model.getGreens(lut[1]);
                model.getBlues(lut[2]);
                model.getAlphas(lut[3]);
            }
            if (out.length() == 0L) {
                int i;
                out.writeBytes("RIFF");
                out.writeInt(0);
                out.writeBytes("AVI ");
                out.writeBytes("LIST");
                out.writeInt(this.bytesPerPixel == 1 ? 1240 : 216);
                out.writeBytes("hdrl");
                out.writeBytes("avih");
                out.writeInt(56);
                this.microSecPerFrame = (int)(1.0 / (double)this.getFramesPerSecond() * 1000000.0);
                out.writeInt(this.microSecPerFrame);
                out.writeInt(0);
                out.writeInt(0);
                out.writeInt(16);
                out.writeInt(0);
                out.writeInt(0);
                out.writeInt(1);
                out.writeInt(0);
                out.writeInt(this.xDim - this.xPad);
                out.writeInt(this.yDim);
                out.writeInt(0);
                out.writeInt(0);
                out.writeInt(0);
                out.writeInt(0);
                out.writeBytes("LIST");
                out.writeInt(this.bytesPerPixel == 1 ? 1164 : 140);
                out.writeBytes("strl");
                out.writeBytes("strh");
                out.writeInt(56);
                out.writeBytes("vids");
                out.writeBytes("DIB ");
                out.writeInt(0);
                out.writeInt(0);
                out.writeInt(0);
                out.writeInt(1);
                out.writeInt(this.getFramesPerSecond());
                out.writeInt(0);
                out.writeInt(this.tDim * this.zDim);
                out.writeInt(0);
                out.writeInt(-1);
                out.writeInt(0);
                out.writeShort(0);
                out.writeShort(0);
                out.writeShort(0);
                out.writeShort(0);
                out.writeBytes("strf");
                out.writeInt(this.bytesPerPixel == 1 ? 1068 : 44);
                out.writeInt(40);
                out.writeInt(this.xDim);
                out.writeInt(this.yDim);
                out.writeShort(1);
                int bitsPerPixel = this.bytesPerPixel == 3 ? 24 : 8;
                out.writeShort((int)((short)bitsPerPixel));
                out.writeInt(0);
                out.writeInt(0);
                out.writeInt(0);
                out.writeInt(0);
                int nColors = 256;
                out.writeInt(256);
                out.writeInt(0);
                if (this.bytesPerPixel == 1) {
                    if (lut != null) {
                        for (i = 0; i < 256; ++i) {
                            out.write((int)lut[2][i]);
                            out.write((int)lut[1][i]);
                            out.write((int)lut[0][i]);
                            out.write((int)lut[3][i]);
                        }
                    } else {
                        byte[] lutWrite = new byte[1024];
                        for (int i2 = 0; i2 < 256; ++i2) {
                            lutWrite[4 * i2] = (byte)i2;
                            lutWrite[4 * i2 + 1] = (byte)i2;
                            lutWrite[4 * i2 + 2] = (byte)i2;
                            lutWrite[4 * i2 + 3] = 0;
                        }
                        out.write(lutWrite);
                    }
                }
                out.seek(168L);
                out.writeInt(1064);
                out.seek(1236L);
                out.writeBytes("strn");
                out.writeInt(16);
                out.writeBytes("FileAVI write  ");
                out.seek(16L);
                out.writeInt(1240);
                out.seek(92L);
                out.writeInt(1164);
                out.seek(1260L);
                out.writeBytes("JUNK");
                out.writeInt(2816);
                i = 0;
                while ((long)i < 1408L) {
                    out.writeShort(0);
                    ++i;
                }
                out.writeBytes("LIST");
                out.writeInt(4);
                out.writeBytes("movi");
                this.idx1Pos = out.offset();
            }
        }
    }

    public static class Reader
    extends ByteArrayReader<Metadata> {
        @Override
        protected String[] createDomainArray() {
            return new String[]{"Graphics"};
        }

        @Override
        public ByteArrayPlane openPlane(int imageIndex, long planeIndex, ByteArrayPlane plane, Interval bounds, SCIFIOConfig config) throws FormatException, IOException {
            byte[] buf = plane.getBytes();
            Metadata meta = (Metadata)this.getMetadata();
            FormatTools.checkPlaneForReading(meta, imageIndex, planeIndex, buf.length, bounds);
            plane.setColorTable(meta.getColorTable(0, 0L));
            int bytes = FormatTools.getBytesPerPixel(meta.get(imageIndex).getPixelType());
            double p = (double)meta.getBmpScanLineSize() / (double)meta.getBmpBitsPerPixel();
            int effectiveWidth = (int)((double)meta.getBmpScanLineSize() / p);
            if (effectiveWidth == 0 || (long)effectiveWidth < meta.get(imageIndex).getAxisLength(Axes.X)) {
                effectiveWidth = (int)meta.get(imageIndex).getAxisLength(Axes.X);
            }
            int xAxis = meta.get(imageIndex).getAxisIndex(Axes.X);
            int yAxis = meta.get(imageIndex).getAxisIndex(Axes.Y);
            int x = (int)bounds.min(xAxis);
            int y = (int)bounds.min(yAxis);
            int w = (int)bounds.dimension(xAxis);
            int h = (int)bounds.dimension(yAxis);
            long fileOff = meta.getOffsets().get((int)planeIndex);
            long end = planeIndex < (long)(meta.getOffsets().size() - 1) ? meta.getOffsets().get((int)planeIndex + 1).longValue() : this.getHandle().length();
            long maxBytes = end - fileOff;
            this.getHandle().seek(fileOff);
            if (meta.getBmpCompression() != 0 && meta.getBmpCompression() != 538982489) {
                this.uncompress(imageIndex, planeIndex, plane, x, y, w, h);
                return plane;
            }
            if (meta.getBmpBitsPerPixel() < 8) {
                int rawSize = (int)FormatTools.getPlaneSize(meta, effectiveWidth, (int)meta.get(imageIndex).getAxisLength(Axes.Y), imageIndex);
                byte[] b = new byte[rawSize /= 8 / meta.getBmpBitsPerPixel()];
                int len = rawSize / (int)meta.get(imageIndex).getAxisLength(Axes.Y);
                this.getHandle().read(b);
                BitBuffer bb = new BitBuffer(b);
                bb.skipBits((long)(meta.getBmpBitsPerPixel() * len) * (meta.get(imageIndex).getAxisLength(Axes.Y) - (long)h - (long)y));
                for (int row = h; row >= y; --row) {
                    bb.skipBits(meta.getBmpBitsPerPixel() * x);
                    for (int col = 0; col < len; ++col) {
                        buf[(row - y) * len + col] = (byte)bb.getBits(meta.getBmpBitsPerPixel());
                    }
                    bb.skipBits((long)meta.getBmpBitsPerPixel() * meta.get(imageIndex).getAxisLength(Axes.X) - (long)w - (long)x);
                }
                return plane;
            }
            int pad = (int)((long)meta.getBmpScanLineSize() / meta.get(imageIndex).getAxisLength(Axes.CHANNEL) - meta.get(imageIndex).getAxisLength(Axes.X) * (long)bytes);
            int scanline = w * bytes * (int)(meta.get(imageIndex).getInterleavedAxisCount() > 0 ? meta.get(imageIndex).getAxisLength(Axes.CHANNEL) : 1L);
            this.getHandle().skipBytes((int)((meta.get(imageIndex).getAxisLength(Axes.X) + (long)pad) * (long)(meta.getBmpBitsPerPixel() / 8) * (meta.get(imageIndex).getAxisLength(Axes.Y) - (long)h - (long)y)));
            if (meta.get(imageIndex).getAxisLength(Axes.X) == (long)w && pad == 0) {
                int row = 0;
                while ((long)row < meta.get(imageIndex).getAxisLength(Axes.Y)) {
                    int outputRow = (int)(meta.getBmpCompression() == 538982489 ? (long)row : meta.get(imageIndex).getAxisLength(Axes.Y) - (long)row - 1L);
                    this.getHandle().read(buf, outputRow * scanline, scanline);
                    ++row;
                }
                if (meta.getBmpBitsPerPixel() == 24 || meta.getBmpBitsPerPixel() == 32) {
                    int i = 0;
                    while ((long)i < (long)buf.length / meta.get(imageIndex).getAxisLength(Axes.CHANNEL)) {
                        byte r = buf[i * (int)meta.get(imageIndex).getAxisLength(Axes.CHANNEL) + 2];
                        buf[i * (int)meta.get((int)imageIndex).getAxisLength((AxisType)Axes.CHANNEL) + 2] = buf[i * (int)meta.get(imageIndex).getAxisLength(Axes.CHANNEL)];
                        buf[i * (int)meta.get((int)imageIndex).getAxisLength((AxisType)Axes.CHANNEL)] = r;
                        ++i;
                    }
                }
            } else {
                int skip = (int)FormatTools.getPlaneSize(meta, (int)meta.get(imageIndex).getAxisLength(Axes.X) - w - x + pad, 1, imageIndex);
                if ((meta.get(imageIndex).getAxisLength(Axes.X) + (long)pad) * meta.get(imageIndex).getAxisLength(Axes.Y) * meta.get(imageIndex).getAxisLength(Axes.CHANNEL) > maxBytes) {
                    skip = (int)((long)skip / meta.get(imageIndex).getAxisLength(Axes.CHANNEL));
                }
                for (int i = h - 1; i >= 0; --i) {
                    this.getHandle().skipBytes(x * (meta.getBmpBitsPerPixel() / 8));
                    this.getHandle().read(buf, i * scanline, scanline);
                    if (meta.getBmpBitsPerPixel() == 24) {
                        for (int j = 0; j < w; ++j) {
                            byte r = buf[i * scanline + j * 3 + 2];
                            buf[i * scanline + j * 3 + 2] = buf[i * scanline + j * 3];
                            buf[i * scanline + j * 3] = r;
                        }
                    }
                    if (i <= 0) continue;
                    this.getHandle().skipBytes(skip);
                }
            }
            if (meta.getBmpBitsPerPixel() == 16 && meta.get(imageIndex).isMultichannel()) {
                ImageTools.bgrToRgb(plane.getBytes(), meta.get(imageIndex).getInterleavedAxisCount() > 0, 2, (int)meta.get(imageIndex).getAxisLength(Axes.CHANNEL));
            }
            return plane;
        }

        private ByteArrayPlane uncompress(int imageIndex, long planeIndex, ByteArrayPlane plane, int x, int y, int w, int h) throws FormatException, IOException {
            Metadata meta = (Metadata)this.getMetadata();
            byte[] buf = null;
            if (this.haveCached(meta, planeIndex, x, y, w, h)) {
                buf = meta.getLastPlane().getBytes();
            } else {
                CodecOptions options = AVIUtils.createCodecOptions(meta, imageIndex, planeIndex);
                FinalInterval bounds = new FinalInterval(meta.get(imageIndex).getAxesLengthsPlanar());
                ByteArrayPlane tmpPlane = this.createPlane((Interval)bounds);
                if (meta.getLastDimensions() != null && !this.sufficientRegion(meta, x, y, w, h)) {
                    long lastPlane = meta.getLastPlaneIndex();
                    meta.setLastDimensions(null);
                    meta.setLastPlane(null);
                    meta.setLastPlaneIndex(-1L);
                    this.openPlane(imageIndex, lastPlane, tmpPlane);
                    options.previousImage = meta.getLastPlaneBytes();
                }
                if (options.previousImage == null && meta.getBmpCompression() != 1196444237) {
                    while (meta.getLastPlaneIndex() < planeIndex - 1L) {
                        this.openPlane(imageIndex, meta.getLastPlaneIndex() + 1L, tmpPlane);
                    }
                    options.previousImage = meta.getLastPlaneBytes();
                }
                buf = AVIUtils.extractCompression(meta, options, this.getHandle(), tmpPlane, planeIndex, new int[]{x, y, w, h});
            }
            int rowLen = (int)FormatTools.getPlaneSize(meta, w, 1, imageIndex);
            int bytes = FormatTools.getBytesPerPixel(meta.get(imageIndex).getPixelType());
            int inputRowLen = (int)FormatTools.getPlaneSize(meta, (int)meta.get(imageIndex).getAxisLength(Axes.X), 1, imageIndex);
            for (int row = 0; row < h; ++row) {
                System.arraycopy(buf, (row + y) * inputRowLen + x * bytes, plane.getBytes(), row * rowLen, rowLen);
            }
            return plane;
        }

        private boolean sufficientRegion(Metadata meta, int x, int y, int w, int h) {
            boolean cached = true;
            int[] dims = meta.getLastDimensions();
            if (dims == null) {
                cached = false;
            } else {
                cached = cached && dims[0] <= x;
                cached = cached && dims[1] <= y;
                cached = cached && dims[2] + dims[0] >= x + w;
                cached = cached && dims[3] + dims[1] >= y + h;
            }
            return cached;
        }

        private boolean haveCached(Metadata meta, long planeIndex, int x, int y, int w, int h) {
            boolean cached = true;
            cached = cached && meta.getLastPlaneIndex() == planeIndex;
            cached = cached && this.sufficientRegion(meta, x, y, w, h);
            return cached;
        }
    }

    public static class Parser
    extends AbstractParser<Metadata> {
        private String type = "error";
        private String fcc = "error";
        private int size = -1;

        @Override
        protected void typedParse(DataHandle<Location> handle, Metadata meta, SCIFIOConfig config) throws IOException, FormatException {
            handle.setOrder(DataHandle.ByteOrder.LITTLE_ENDIAN);
            this.log().info((Object)"Verifying AVI format");
            meta.setLastPlaneIndex(-1L);
            meta.setLengths(new ArrayList<Long>());
            meta.setOffsets(new ArrayList<Long>());
            meta.createImageMetadata(1);
            while (handle.offset() < handle.length() - 8L) {
                this.readChunk(meta);
            }
            this.log().info((Object)"Populating metadata");
        }

        private void readChunkHeader() throws IOException {
            this.readTypeAndSize();
            this.fcc = this.getSource().readString(4);
        }

        private void readTypeAndSize() throws IOException {
            this.type = this.getSource().readString(4);
            this.size = this.getSource().readInt();
        }

        private void readChunk(Metadata meta) throws FormatException, IOException {
            long pos;
            this.readChunkHeader();
            ImageMetadata m = meta.get(0);
            MetaTable globalTable = meta.getTable();
            DataHandle<Location> source = this.getSource();
            if (this.type.equals("RIFF")) {
                if (!this.fcc.startsWith("AVI")) {
                    throw new FormatException("Sorry, AVI RIFF format not found.");
                }
            } else {
                if (source.offset() == 12L) {
                    throw new FormatException("Not an AVI file");
                }
                if (source.offset() + (long)this.size - 4L <= source.length()) {
                    source.skipBytes(this.size - 4);
                }
                return;
            }
            long spos = pos = source.offset();
            this.log().info((Object)"Searching for image data");
            while (source.length() - source.offset() > 4L) {
                String listString = source.readString(4);
                if (listString.equals("RIFF")) {
                    source.seek(source.offset() - 4L);
                    return;
                }
                source.seek(pos);
                if (listString.equals(" JUN")) {
                    source.skipBytes(1);
                    ++pos;
                }
                if (listString.equals("JUNK")) {
                    this.readTypeAndSize();
                    if (this.type.equals("JUNK")) {
                        source.skipBytes(this.size);
                    }
                } else if (listString.equals("LIST")) {
                    spos = source.offset();
                    this.readChunkHeader();
                    source.seek(spos);
                    if (this.fcc.equals("hdrl")) {
                        this.readChunkHeader();
                        if (this.type.equals("LIST") && this.fcc.equals("hdrl")) {
                            this.readTypeAndSize();
                            if (this.type.equals("avih")) {
                                spos = source.offset();
                                globalTable.put("Microseconds per frame", source.readInt());
                                globalTable.put("Max. bytes per second", source.readInt());
                                source.skipBytes(8);
                                globalTable.put("Total frames", source.readInt());
                                globalTable.put("Initial frames", source.readInt());
                                source.skipBytes(8);
                                m.addAxis(Axes.X, (long)source.readInt());
                                globalTable.put("Frame height", source.readInt());
                                globalTable.put("Scale factor", source.readInt());
                                globalTable.put("Frame rate", source.readInt());
                                globalTable.put("Start time", source.readInt());
                                globalTable.put("Length", source.readInt());
                                globalTable.put("Frame width", m.getAxisLength(Axes.X));
                                if (spos + (long)this.size <= source.length()) {
                                    source.seek(spos + (long)this.size);
                                }
                            }
                        }
                    } else if (this.fcc.equals("strl")) {
                        long startPos = source.offset();
                        long streamSize = this.size;
                        this.readChunkHeader();
                        if (this.type.equals("LIST")) {
                            if (this.fcc.equals("strl")) {
                                this.readTypeAndSize();
                                if (this.type.equals("strh")) {
                                    spos = source.offset();
                                    source.skipBytes(40);
                                    globalTable.put("Stream quality", source.readInt());
                                    meta.setBytesPerPlane(source.readInt());
                                    globalTable.put("Stream sample size", meta.getBytesPerPlane());
                                    if (spos + (long)this.size <= source.length()) {
                                        source.seek(spos + (long)this.size);
                                    }
                                }
                                this.readTypeAndSize();
                                if (this.type.equals("strf")) {
                                    spos = source.offset();
                                    if (meta.get(0).getAxisIndex(Axes.Y) != -1) {
                                        source.skipBytes(this.size);
                                        this.readTypeAndSize();
                                        while (this.type.equals("indx")) {
                                            source.skipBytes(this.size);
                                            this.readTypeAndSize();
                                        }
                                        pos = source.offset() - 4L;
                                        source.seek(pos - 4L);
                                        continue;
                                    }
                                    source.skipBytes(4);
                                    meta.setBmpWidth(source.readInt());
                                    m.addAxis(Axes.Y, (long)source.readInt());
                                    source.skipBytes(2);
                                    meta.setBmpBitsPerPixel(source.readShort());
                                    meta.setBmpCompression(source.readInt());
                                    source.skipBytes(4);
                                    globalTable.put("Horizontal resolution", source.readInt());
                                    globalTable.put("Vertical resolution", source.readInt());
                                    meta.setBmpColorsUsed(source.readInt());
                                    source.skipBytes(4);
                                    globalTable.put("Bitmap compression value", meta.getBmpCompression());
                                    globalTable.put("Number of colors used", meta.getBmpColorsUsed());
                                    globalTable.put("Bits per pixel", meta.getBmpBitsPerPixel());
                                    int npad = meta.getBmpWidth() % 4;
                                    if (npad > 0) {
                                        npad = 4 - npad;
                                    }
                                    meta.setBmpScanLineSize((meta.getBmpWidth() + npad) * (meta.getBmpBitsPerPixel() / 8));
                                    int bmpActualColorsUsed = 0;
                                    if (meta.getBmpColorsUsed() != 0) {
                                        bmpActualColorsUsed = meta.getBmpColorsUsed();
                                    } else if (meta.getBmpBitsPerPixel() < 16) {
                                        bmpActualColorsUsed = 1 << meta.getBmpBitsPerPixel();
                                        meta.setBmpColorsUsed(bmpActualColorsUsed);
                                    }
                                    if (meta.getBmpCompression() != 1 && meta.getBmpCompression() != 0 && meta.getBmpCompression() != 1296126531 && meta.getBmpCompression() != 1196444237 && meta.getBmpCompression() != 538982489) {
                                        throw new UnsupportedCompressionException(meta.getBmpCompression() + " not supported");
                                    }
                                    if (meta.getBmpBitsPerPixel() != 4 && meta.getBmpBitsPerPixel() != 8 && meta.getBmpBitsPerPixel() != 24 && meta.getBmpBitsPerPixel() != 16 && meta.getBmpBitsPerPixel() != 32) {
                                        throw new FormatException(meta.getBmpBitsPerPixel() + " bits per pixel not supported");
                                    }
                                    if (bmpActualColorsUsed != 0) {
                                        byte[][] lut = new byte[3][meta.getBmpColorsUsed()];
                                        for (int i = 0; i < meta.getBmpColorsUsed(); ++i) {
                                            if (meta.getBmpCompression() != 538982489) {
                                                lut[2][i] = source.readByte();
                                                lut[1][i] = source.readByte();
                                                lut[0][i] = source.readByte();
                                                source.skipBytes(1);
                                                continue;
                                            }
                                            lut[0][i] = (byte)i;
                                            lut[1][i] = (byte)i;
                                            lut[2][i] = (byte)i;
                                        }
                                        meta.lut = (ColorTable)new ColorTable8((byte[][])new byte[][]{lut[0], lut[1], lut[2]});
                                    }
                                    source.seek(spos + (long)this.size);
                                }
                            }
                            spos = source.offset();
                            this.readTypeAndSize();
                            if (this.type.equals("strd")) {
                                source.skipBytes(this.size);
                            } else {
                                source.seek(spos);
                            }
                            spos = source.offset();
                            this.readTypeAndSize();
                            if (this.type.equals("strn") || this.type.equals("indx")) {
                                source.skipBytes(this.size);
                            } else {
                                source.seek(spos);
                            }
                        }
                        if (startPos + streamSize + 8L <= source.length()) {
                            source.seek(startPos + 8L + streamSize);
                        }
                    } else if (this.fcc.equals("movi")) {
                        this.readChunkHeader();
                        if (this.type.equals("LIST") && this.fcc.equals("movi")) {
                            spos = source.offset();
                            if (spos >= source.length() - 12L) break;
                            this.readChunkHeader();
                            if (!this.type.equals("LIST") || !this.fcc.equals("rec ") && !this.fcc.equals("movi")) {
                                source.seek(spos);
                            }
                            spos = source.offset();
                            boolean end = false;
                            while (!end) {
                                this.readTypeAndSize();
                                String oldType = this.type;
                                while (this.type.startsWith("ix") || this.type.endsWith("tx") || this.type.equals("JUNK")) {
                                    source.skipBytes(this.size);
                                    this.readTypeAndSize();
                                }
                                String check = this.type.substring(2);
                                boolean foundPixels = false;
                                while (check.equals("db") || check.equals("dc") || check.equals("wb")) {
                                    foundPixels = true;
                                    if (check.startsWith("d") && (this.size > 0 || meta.getBmpCompression() != 0)) {
                                        meta.getOffsets().add(source.offset());
                                        meta.getLengths().add(Long.valueOf(this.size));
                                        source.skipBytes(this.size);
                                    }
                                    if ((spos = source.offset()) + 8L >= source.length()) {
                                        return;
                                    }
                                    this.readTypeAndSize();
                                    if (this.type.equals("JUNK")) {
                                        source.skipBytes(this.size);
                                        spos = source.offset();
                                        if (spos + 8L >= source.length()) {
                                            return;
                                        }
                                        this.readTypeAndSize();
                                    }
                                    if (!(check = this.type.substring(2)).equals("0d")) continue;
                                    source.seek(spos + 1L);
                                    this.readTypeAndSize();
                                    check = this.type.substring(2);
                                }
                                source.seek(spos);
                                if (oldType.startsWith("ix") || foundPixels) continue;
                                end = true;
                            }
                        }
                    } else {
                        int oldSize = this.size;
                        this.size = source.readInt() - 8;
                        if (this.size > oldSize) {
                            this.size = oldSize;
                            source.seek(source.offset() - 4L);
                        }
                        if (this.size + 8 >= 0) {
                            source.skipBytes(8 + this.size);
                        }
                    }
                } else {
                    this.readTypeAndSize();
                    if (source.offset() + 8L < source.length() && !this.type.equals("idx1")) {
                        this.readTypeAndSize();
                    } else if (!this.type.equals("idx1")) break;
                    if (source.offset() + (long)this.size + 4L <= source.length()) {
                        source.skipBytes(this.size);
                    }
                    if (this.type.equals("idx1")) break;
                }
                pos = source.offset();
            }
        }
    }

    public static class Checker
    extends AbstractChecker {
        public static final String AVI_MAGIC_STRING = "RIFF";

        @Override
        public boolean suffixNecessary() {
            return false;
        }

        @Override
        public boolean isFormat(DataHandle<Location> handle) throws IOException {
            int blockLen = 12;
            if (!FormatTools.validStream(handle, 12, false)) {
                return false;
            }
            String type = handle.readString(4);
            handle.skipBytes(4);
            String format = handle.readString(4);
            return type.equals(AVI_MAGIC_STRING) && format.equals("AVI ");
        }
    }

    public static class Metadata
    extends AbstractMetadata
    implements HasColorTable {
        private List<Long> offsets;
        private List<Long> lengths;
        private short bmpBitsPerPixel;
        private int bmpCompression;
        private int bmpScanLineSize;
        private int bmpColorsUsed;
        private int bmpWidth;
        private int bytesPerPlane;
        private ColorTable lut;
        private ByteArrayPlane lastPlane;
        private long lastPlaneIndex;
        private int[] lastDims;

        public short getBmpBitsPerPixel() {
            return this.bmpBitsPerPixel;
        }

        public void setBmpBitsPerPixel(short bmpBitsPerPixel) {
            this.bmpBitsPerPixel = bmpBitsPerPixel;
        }

        public int getBmpCompression() {
            return this.bmpCompression;
        }

        public void setBmpCompression(int bmpCompression) {
            this.bmpCompression = bmpCompression;
        }

        public int getBmpScanLineSize() {
            return this.bmpScanLineSize;
        }

        public void setBmpScanLineSize(int bmpScanLineSize) {
            this.bmpScanLineSize = bmpScanLineSize;
        }

        public List<Long> getOffsets() {
            return this.offsets;
        }

        public void setOffsets(List<Long> offsets) {
            this.offsets = offsets;
        }

        public List<Long> getLengths() {
            return this.lengths;
        }

        public void setLengths(List<Long> lengths) {
            this.lengths = lengths;
        }

        public int getBmpColorsUsed() {
            return this.bmpColorsUsed;
        }

        public void setBmpColorsUsed(int bmpColorsUsed) {
            this.bmpColorsUsed = bmpColorsUsed;
        }

        public int getBmpWidth() {
            return this.bmpWidth;
        }

        public void setBmpWidth(int bmpWidth) {
            this.bmpWidth = bmpWidth;
        }

        public ByteArrayPlane getLastPlane() {
            return this.lastPlane;
        }

        public byte[] getLastPlaneBytes() {
            return this.lastPlane == null ? null : this.lastPlane.getBytes();
        }

        public void setLastDimensions(int[] dims) {
            this.lastDims = dims;
        }

        public int[] getLastDimensions() {
            return this.lastDims;
        }

        public void setLastPlane(ByteArrayPlane lastPlane) {
            this.lastPlane = lastPlane;
        }

        public long getLastPlaneIndex() {
            return this.lastPlaneIndex;
        }

        public void setLastPlaneIndex(long lastPlaneIndex) {
            this.lastPlaneIndex = lastPlaneIndex;
        }

        public int getBytesPerPlane() {
            return this.bytesPerPlane;
        }

        public void setBytesPerPlane(int bytesPerPlane) {
            this.bytesPerPlane = bytesPerPlane;
        }

        @Override
        public ColorTable getColorTable(int imageIndex, long planeIndex) {
            return this.lut;
        }

        @Override
        public void populateImageMetadata() {
            int effectiveWidth;
            ImageMetadata iMeta = this.get(0);
            iMeta.setLittleEndian(true);
            iMeta.setFalseColor(false);
            iMeta.setMetadataComplete(true);
            int sizeT = this.getOffsets().size();
            this.getTable().put("Compression", AVIUtils.getCodecName(this.getBmpCompression()));
            iMeta.setPlanarAxisCount(2);
            if (this.getBmpCompression() == 1196444237) {
                long fileOff = this.getOffsets().get(0);
                CodecOptions options = AVIUtils.createCodecOptions(this, 0, 0L);
                int nBytes = 0;
                try {
                    boolean x = false;
                    boolean y = false;
                    int w = (int)iMeta.getAxisLength(Axes.X);
                    int h = (int)iMeta.getAxisLength(Axes.Y);
                    nBytes = AVIUtils.extractCompression(this, options, this.getSource(), null, 0L, new int[]{0, 0, w, h}).length / (w * h);
                }
                catch (IOException e) {
                    this.log().error((Object)"IOException while decompressing", (Throwable)e);
                }
                catch (FormatException e) {
                    this.log().error((Object)"FormatException while decompressing", (Throwable)e);
                }
                try {
                    this.getSource().seek(fileOff);
                }
                catch (IOException e) {
                    this.log().error((Object)("Error seeking to position: " + fileOff), (Throwable)e);
                }
                if (this.getBmpCompression() == 16) {
                    nBytes /= 2;
                }
                if (nBytes > 1) {
                    iMeta.addAxis(Axes.CHANNEL, (long)nBytes);
                    iMeta.setPlanarAxisCount(3);
                }
            } else if (this.getBmpBitsPerPixel() == 32) {
                iMeta.addAxis(Axes.CHANNEL, 4L);
                iMeta.setPlanarAxisCount(3);
            } else if (this.getBytesPerPlane() == 0 || this.getBmpBitsPerPixel() == 24) {
                if (this.getBmpBitsPerPixel() > 8 || this.getBmpCompression() != 0 && this.getColorTable(0, 0L) == null) {
                    iMeta.addAxis(Axes.CHANNEL, 3L);
                    iMeta.setPlanarAxisCount(3);
                }
            } else if (this.getBmpCompression() == 1296126531) {
                iMeta.addAxis(Axes.CHANNEL, 3L);
                iMeta.setPlanarAxisCount(3);
            } else {
                long sizeC = (long)this.getBytesPerPlane() / (iMeta.getAxisLength(Axes.X) * iMeta.getAxisLength(Axes.Y) * (long)(this.getBmpBitsPerPixel() / 8));
                if (sizeC > 1L) {
                    iMeta.addAxis(Axes.CHANNEL, sizeC);
                    iMeta.setPlanarAxisCount(3);
                }
            }
            if (this.getColorTable(0, 0L) != null && !iMeta.isMultichannel()) {
                iMeta.setIndexed(true);
                iMeta.addAxis(Axes.CHANNEL, 1L);
                iMeta.setPlanarAxisCount(3);
                iMeta.setAxisType(2, Axes.CHANNEL);
            }
            if (this.getBmpBitsPerPixel() <= 8) {
                iMeta.setPixelType(1);
            } else if (this.getBmpBitsPerPixel() == 16) {
                iMeta.setPixelType(3);
            } else if (this.getBmpBitsPerPixel() == 24 || this.getBmpBitsPerPixel() == 32) {
                iMeta.setPixelType(1);
            } else {
                this.log().error((Object)("Unknown matching for pixel bit width of: " + this.getBmpBitsPerPixel()));
            }
            if (this.getBmpCompression() != 0) {
                iMeta.setPixelType(1);
            }
            if ((effectiveWidth = this.getBmpScanLineSize() / (this.getBmpBitsPerPixel() / 8)) == 0) {
                effectiveWidth = (int)iMeta.getAxisLength(Axes.X);
            }
            if ((long)effectiveWidth < iMeta.getAxisLength(Axes.X)) {
                iMeta.setAxisLength(Axes.X, (long)effectiveWidth);
            }
            if (this.getBmpBitsPerPixel() != 16 && iMeta.isMultichannel()) {
                iMeta.setAxisTypes(Axes.CHANNEL, Axes.X, Axes.Y);
            }
            iMeta.addAxis(Axes.TIME, (long)sizeT);
        }

        @Override
        public void close(boolean fileOnly) throws IOException {
            super.close(fileOnly);
            if (!fileOnly) {
                this.lastPlane = null;
                this.lastPlaneIndex = -1L;
                this.bmpScanLineSize = 0;
                this.bmpCompression = 0;
                this.bmpWidth = 0;
                this.bmpColorsUsed = 0;
                this.bmpBitsPerPixel = 0;
                this.bytesPerPlane = 0;
                this.offsets = null;
                this.lengths = null;
                this.lut = null;
            }
        }
    }
}

